import {FunctionComponent, ReactNode, useRef, useCallback} from 'react';

import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import {
    EHotelBadgeType,
    IHotelOrOfferBadge,
} from 'types/hotels/offer/IHotelOffer';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {params} from 'utilities/metrika';
import {internalUrl} from 'utilities/url';
import {deviceMods} from 'utilities/stylesUtils';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import getLabelTheme from './utilities/getLabelTheme';
import getText from './utilities/getText';
import getIcon from './utilities/getIcon';

import * as i18nBlock from 'i18n/mirCashback';

import Box from 'components/Box/Box';
import Link from 'components/Link/Link';
import Label from 'components/Label/Label';
import Heading from 'components/Heading/Heading';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import {EMessageBoxPopupTheme} from 'components/MessageBoxPopup/MessageBoxPopup';
import MessageBoxPopupOrBottomSheet from 'components/MessageBoxPopupOrBottomSheet/MessageBoxPopupOrBottomSheet';

import ScopeContext from 'contexts/ScopeContext';

import cx from './HotelOfferBadge.scss';

const MIR_ICON_SIZE = {width: 27};

const POPUP_POSITIONS = [EPopupDirection.BOTTOM, EPopupDirection.TOP];

export interface IHotelOfferBadgeProps extends IWithClassName {
    badge: IHotelOrOfferBadge;
}

const HotelOfferBadge: FunctionComponent<IHotelOfferBadgeProps> = ({
    className,
    badge,
}) => {
    const icon = getIcon(badge.id);
    const text = getText(badge);

    const isMirCashback = badge.id === EHotelBadgeType.MIR_CASHBACK;

    const deviceType = useDeviceType();
    const labelRef = useRef<HTMLSpanElement>(null);
    const {
        value: isVisible,
        setTrue: setVisible,
        setFalse: setHidden,
    } = useBoolean(false);

    const handleBadgeClick = useCallback(() => {
        setVisible();
        params({
            hotels: {mir: {badgeClick: true}},
        });
    }, [setVisible]);

    const handlePopupLinkClick = useCallback(() => {
        setVisible();
        params({
            hotels: {mir: {popupLinkClick: true}},
        });
    }, [setVisible]);

    return (
        <>
            <Label
                className={cx(isMirCashback ? 'mirCashback' : '', className)}
                theme={getLabelTheme(badge.id)}
                ref={labelRef}
                onClick={handleBadgeClick}
            >
                {icon ? (
                    <TextWithIcon
                        className={cx('textWithIcon')}
                        textClassName={cx('text')}
                        size="xs"
                        text={text}
                        iconLeft={isMirCashback ? undefined : icon}
                        iconRight={isMirCashback ? icon : undefined}
                        iconRightProps={
                            isMirCashback ? MIR_ICON_SIZE : undefined
                        }
                    />
                ) : (
                    text
                )}
            </Label>
            {isMirCashback && (
                <ScopeContext.Consumer>
                    {(scopeRef): ReactNode => (
                        <MessageBoxPopupOrBottomSheet
                            messageBoxProps={{
                                theme: EMessageBoxPopupTheme.WHITE,
                                anchorRef: labelRef,
                                scopeRef: scopeRef
                                    ? {current: scopeRef}
                                    : undefined,
                                direction: POPUP_POSITIONS,
                            }}
                            isVisible={isVisible}
                            onClose={setHidden}
                        >
                            <div
                                className={cx(
                                    deviceMods('popupContent', deviceType),
                                )}
                            >
                                <Heading level={3}>
                                    {i18nBlock.popupTitle()}
                                </Heading>
                                <Box above="2" below="2" textSize="m">
                                    {i18nBlock.popupText()}
                                </Box>
                                <div className={cx('popupLink')}>
                                    <Link
                                        url={internalUrl(URLs.mirCashback)}
                                        target="_blank"
                                        rel="noopener noreferrer"
                                        onClick={handlePopupLinkClick}
                                    >
                                        {i18nBlock.popupLink()}
                                    </Link>
                                </div>
                            </div>
                        </MessageBoxPopupOrBottomSheet>
                    )}
                </ScopeContext.Consumer>
            )}
        </>
    );
};

export default HotelOfferBadge;
