import React, {useCallback} from 'react';

import {BASE_LINK_PROPS} from 'constants/linkProps';

import {IHotel} from 'types/hotels/hotel/IHotel';

import {reachGoal} from 'utilities/metrika';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {CHAR_MIDDLE_DOT, CHAR_NBSP} from 'utilities/strings/charCodes';
import getHotelReviewsGoalByPage from './utilities/getHotelReviewsGoalByPage';

import * as i18nBlock from 'i18n/hotels-SearchHotelCard';

import Link from 'components/Link/Link';
import {ECardUsagePage} from 'projects/depreacted/hotels/components/HotelInfoSearchCard/HotelInfoSearchCard';
import Rating from 'components/Rating/Rating';

import cx from './HotelRatingAndCategory.scss';

interface IHotelRatingAndCategoryProps {
    hotel: IHotel;
    hotelReviewsUrl: string;
    cardUsagePage?: ECardUsagePage;
    size: 's' | 'm';
}

const HotelRatingAndCategory: React.FC<IHotelRatingAndCategoryProps> =
    props => {
        const {
            hotel: {
                rating,
                category: {name: categoryName},
                totalTextReviewCount,
            },
            hotelReviewsUrl,
            cardUsagePage,
            size,
        } = props;

        const {isMobile} = useDeviceType();

        const handleHotelReviewsClick = useCallback(() => {
            const goal =
                cardUsagePage && getHotelReviewsGoalByPage(cardUsagePage);

            if (goal) {
                reachGoal(goal);
            }
        }, [cardUsagePage]);
        const hotelReviewsUrlProps = isMobile
            ? {
                  to: hotelReviewsUrl,
                  target: '_self',
                  ...BASE_LINK_PROPS,
              }
            : {
                  url: hotelReviewsUrl,
                  target: '_blank',
                  ...BASE_LINK_PROPS,
              };

        return (
            <div
                className={cx('ratingsInfo', `ratingsInfo_${size}`, {
                    ratingsInfo_mobile: isMobile,
                })}
            >
                {Boolean(rating) && (
                    <Rating
                        size={size}
                        className={cx('rating')}
                        rating={rating}
                    />
                )}
                {Boolean(totalTextReviewCount) && (
                    <span className={cx('reviewsCount')}>
                        <Link
                            className={cx('reviewsCountLink')}
                            theme="black"
                            onClick={handleHotelReviewsClick}
                            {...hotelReviewsUrlProps}
                        >
                            {i18nBlock.reviewsCount({
                                reviewsCount: totalTextReviewCount,
                            })}
                        </Link>
                        {CHAR_NBSP}
                        {CHAR_NBSP}
                        {CHAR_MIDDLE_DOT}
                        {CHAR_NBSP}
                        {CHAR_NBSP}
                    </span>
                )}
                {Boolean(categoryName) && (
                    <span className={cx('categoryName')}>{categoryName}</span>
                )}
            </div>
        );
    };

export default HotelRatingAndCategory;
