import {PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IKeyPhrase, ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {IHotelReviewReactionPayload} from 'reducers/depreacted/hotels/hotelPage/reviews/list/types';
import {
    ETextReviewRankingType,
    isETextReviewRankingType,
} from 'server/api/HotelSearchAPI/types/ETextReviewRankingType';
import {TSortId, ISortInfo, ESortTypeHint} from 'types/hotels/search/ISortInfo';

import debounceById from 'utilities/functions/debounceById';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/hotels-HotelReviews';
import * as i18nCommonBlock from 'i18n/hotels-Common';
import * as i18nReviewsSortingOptionsBlock from 'i18n/hotels-HotelReviewsSortingOptions';

import Card from 'components/Card/Card';
import LinkButton from 'components/LinkButton/LinkButton';
import Separator from 'components/Separator/Separator';
import Spinner from 'components/Spinner/Spinner';
import LazyLoad from 'projects/depreacted/hotels/components/LazyLoad/LazyLoad';
import TextHotelReviewsSkeleton from './components/TextHotelReviewsSkeleton/TextHotelReviewsSkeleton';
import HotelReviewWithReaction from './components/TextHotelReview/HotelReviewWithReaction';
import KeyPhraseCheckButton from './components/KeyPhraseCheckButton/KeyPhraseCheckButton';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import HotelsSort from 'projects/depreacted/hotels/components/HotelsSort/HotelsSort';
import KebabVerticalIcon from 'icons/16/KebabVertical';

import {IHotelReviewsContainer} from './HotelReviewsContainer';

/* init Styles */
import cx from './HotelReviews.scss';

/* Constants */
const REVIEW_PLACEHOLDERS_COUNT = 1;
const SET_REVIEW_REACTION_DEBOUNCE = 2000;
const REVIEWS_LOAD_COUNT = 10;

const REVIEWS_SORTING_INFO: ISortInfo = {
    availableSortTypeGroups: [
        {
            id: ETextReviewRankingType.RELEVANCE_ORG,
            name: i18nReviewsSortingOptionsBlock.byRelevanceOrg(),
            requiresGeoLocation: false,
            sortTypes: [
                {
                    id: ETextReviewRankingType.RELEVANCE_ORG,
                    name: i18nReviewsSortingOptionsBlock.byRelevanceOrg(),
                    hint: ESortTypeHint.descending,
                },
            ],
        },
        {
            id: ETextReviewRankingType.TIME,
            name: i18nReviewsSortingOptionsBlock.byTime(),
            requiresGeoLocation: false,
            sortTypes: [
                {
                    id: ETextReviewRankingType.TIME,
                    name: i18nReviewsSortingOptionsBlock.byTime(),
                    hint: ESortTypeHint.descending,
                },
            ],
        },
        {
            id: ETextReviewRankingType.RATING_ASC,
            name: i18nReviewsSortingOptionsBlock.byRatingAsc(),
            requiresGeoLocation: false,
            sortTypes: [
                {
                    id: ETextReviewRankingType.RATING_ASC,
                    name: i18nReviewsSortingOptionsBlock.byRatingAsc(),
                    hint: ESortTypeHint.descending,
                },
            ],
        },
        {
            id: ETextReviewRankingType.RATING_DESC,
            name: i18nReviewsSortingOptionsBlock.byRatingDesc(),
            requiresGeoLocation: false,
            sortTypes: [
                {
                    id: ETextReviewRankingType.RATING_DESC,
                    name: i18nReviewsSortingOptionsBlock.byRatingDesc(),
                    hint: ESortTypeHint.descending,
                },
            ],
        },
    ],
};

/* Component Types */
interface IHotelReviewsProps
    extends IHotelReviewsContainer,
        IWithClassName,
        IWithDeviceType {
    isActiveTab: boolean;
    isLazyLoading?: boolean;
    activeKeyPhrase?: string;
    maxReviewRenderCount?: number;
    totalReviewCount?: number;

    onMoreReviews?: () => void;
    onKeyPhraseClick?: (keyPhrase: string) => void;
}

interface IHotelReviewsState {
    isAllKeyPhrases: boolean;
}

class HotelReviews extends PureComponent<
    IHotelReviewsProps,
    IHotelReviewsState
> {
    state = {
        isAllKeyPhrases: false,
    };

    /* Helpers */

    private debouncedReactionRequest = debounceById(
        (reactionPayload: IHotelReviewReactionPayload) => {
            const {setHotelReviewReactionRequest} = this.props;

            setHotelReviewReactionRequest(reactionPayload);
        },
        SET_REVIEW_REACTION_DEBOUNCE,
    );

    /* Handlers */

    private handleKeyPhraseClick = (keyPhrase: string): void => {
        const {onKeyPhraseClick} = this.props;

        if (onKeyPhraseClick) {
            onKeyPhraseClick(keyPhrase);
        }
    };

    private handleReviewsButtonClick = (): void => {
        const {onMoreReviews} = this.props;

        if (onMoreReviews) {
            onMoreReviews();
        }
    };

    private handleSetReviewReaction = (
        reactionPayload: IHotelReviewReactionPayload,
    ): void => {
        const {setHotelReviewReaction} = this.props;

        setHotelReviewReaction(reactionPayload);
        this.debouncedReactionRequest(
            reactionPayload.reviewId,
            reactionPayload,
        );
    };

    private showKeyPhrases = (): void => {
        this.setState({
            isAllKeyPhrases: true,
        });
    };

    private handleReviewsSortingOptionsChange = ({id}: {id: TSortId}): void => {
        const {
            hotelReviews: {isLoading, activeKeyPhrase},
            sortingHotelReviews,
        } = this.props;

        if (isETextReviewRankingType(id) && !isLoading) {
            sortingHotelReviews({
                textReviewOffset: 0,
                textReviewLimit: REVIEWS_LOAD_COUNT,
                keyPhraseFilter: activeKeyPhrase,
                textReviewRanking: id,
            });
        }
    };

    /* Render */

    private renderReviewTotalCount(): ReactNode {
        const {hotelReviews} = this.props;

        const {
            data: {totalTextReviewCount},
        } = hotelReviews;

        return (
            <Heading
                level={2}
                className={cx('totalReviewCount')}
                {...prepareQaAttributes('hotelPageReviewsTitle')}
            >
                {i18nCommonBlock.totalReviewCount({
                    totalReviewCount: totalTextReviewCount,
                })}
            </Heading>
        );
    }

    private renderTextReview = (textReview: ITextReview): ReactNode => {
        const {
            userInfo,
            deviceType: {isMobile},
        } = this.props;

        const textHotelReviewNode = (
            <HotelReviewWithReaction
                className={cx('textReview')}
                key={textReview.id}
                isAuth={isAuthUser(userInfo)}
                review={textReview}
                onHotelReviewReaction={this.handleSetReviewReaction}
            />
        );

        return isMobile ? (
            <Card shadow="default" key={textReview.id}>
                {textHotelReviewNode}
            </Card>
        ) : (
            textHotelReviewNode
        );
    };

    private renderLoader(): ReactNode {
        const {
            deviceType: {isMobile},
        } = this.props;

        return (
            <Spinner
                className={cx('loader')}
                size={isMobile ? 'm' : 'xxs'}
                {...prepareQaAttributes('moreReviewsLoader')}
            />
        );
    }

    private renderTextReviews(): ReactNode {
        const {
            hotelReviews,
            isLazyLoading,
            onMoreReviews,
            maxReviewRenderCount,
        } = this.props;

        const {
            isLoading,
            isLoadingList,
            data: {textReviews},
        } = hotelReviews;

        const resultTextReviews = maxReviewRenderCount
            ? textReviews.slice(0, maxReviewRenderCount)
            : textReviews;

        return (
            <div className={cx('textReviews')}>
                {isLazyLoading && onMoreReviews ? (
                    <LazyLoad
                        isLoading={isLoading || isLoadingList}
                        placeholdersCount={REVIEW_PLACEHOLDERS_COUNT}
                        placeholderNode={this.renderLoader()}
                        onBottomScroll={onMoreReviews}
                    >
                        {resultTextReviews.map(this.renderTextReview)}
                    </LazyLoad>
                ) : (
                    resultTextReviews.map(this.renderTextReview)
                )}
            </div>
        );
    }

    private renderSortingOptions(): ReactNode {
        const {
            deviceType,
            hotelReviews: {currentSortingOption},
        } = this.props;

        const sortInfo = {
            ...REVIEWS_SORTING_INFO,
            selectedSortId: currentSortingOption,
        };

        return (
            <HotelsSort
                sortInfo={sortInfo}
                deviceType={deviceType}
                needSyncSortWithServer={false}
                setActiveSort={this.handleReviewsSortingOptionsChange}
                size="m"
                showIcon={deviceType.isMobile}
                className={cx('hotelsSort')}
            />
        );
    }

    /**
     * Сейчас ключевые слова приходят не валидные,
     * Вернуть когда бэк разрешит проблемы с ними
     */
    private renderKeyPhrases(): ReactNode {
        return null;
        // const {
        //     reviewsInfo: {keyPhrases},
        //     deviceType: {isMobile}
        // } = this.props;
        /** Вынести "7" в константы с возвратом кнопкок */
        // const {isAllKeyPhrases} = this.state;
        // const canRenderAll = keyPhrases.length < 7 ||
        //     isMobile ||
        //     isAllKeyPhrases;
        // const visibleKeyPhrases = canRenderAll
        //     ? keyPhrases
        //     : keyPhrases.slice(0, 7);
        // return (
        //     <>
        //         <div className={cx('keyPhrases')}>
        //             {visibleKeyPhrases.map(this.renderKeyPhrase)}
        //             {!canRenderAll && this.renderKeyPhraseToggle()}
        //         </div>
        //     </>
        // );
    }

    renderKeyPhraseToggle(): ReactNode {
        return (
            <Button
                className={cx('keyPhrase')}
                size="s"
                onClick={this.showKeyPhrases}
            >
                <KebabVerticalIcon />
            </Button>
        );
    }

    /**
     * TODO: убрать когда вернутся ключевые слова
     */
    renderKeyPhrase = (keyPhrase: IKeyPhrase): ReactNode => {
        const {activeKeyPhrase} = this.props;

        return (
            <KeyPhraseCheckButton
                className={cx('keyPhrase')}
                keyPhrase={keyPhrase}
                key={keyPhrase.name}
                checked={keyPhrase.name === activeKeyPhrase}
                onClick={this.handleKeyPhraseClick}
            />
        );
    };

    private renderMoreReviewsButton(): ReactNode {
        const {isActiveTab, hotelReviews, maxReviewRenderCount} = this.props;
        const {
            data: {textReviews, totalTextReviewCount},
            isLoading,
        } = hotelReviews;
        const loadReviewCount =
            totalTextReviewCount - (maxReviewRenderCount || textReviews.length);
        const moreReviewCount = isActiveTab
            ? Math.min(REVIEWS_LOAD_COUNT, loadReviewCount)
            : loadReviewCount;

        if (moreReviewCount > 0) {
            return (
                <div className={cx('moreReviewsButton')}>
                    <LinkButton
                        disabled={isLoading}
                        onClick={this.handleReviewsButtonClick}
                        {...prepareQaAttributes('moreReviewsButton')}
                    >
                        {i18nBlock.otherReviews({
                            totalReviewCount: moreReviewCount,
                        })}
                    </LinkButton>
                    {isLoading && this.renderLoader()}
                </div>
            );
        }

        return null;
    }

    private renderSkeleton(hideKeyPhrases: boolean): ReactNode {
        const {deviceType} = this.props;

        return (
            <TextHotelReviewsSkeleton
                deviceType={deviceType}
                hideKeyPhrases={hideKeyPhrases}
                {...prepareQaAttributes('hotelReviewsSkeleton')}
            />
        );
    }

    private renderDesktop = (): ReactNode => {
        const {
            isLazyLoading,
            hotelReviews: {isLoadingList},
        } = this.props;

        return (
            <>
                {this.renderReviewTotalCount()}
                {this.renderSortingOptions()}
                {this.renderKeyPhrases()}
                <Separator />
                {isLoadingList
                    ? this.renderSkeleton(true)
                    : this.renderTextReviews()}
                {!isLazyLoading &&
                    !isLoadingList &&
                    this.renderMoreReviewsButton()}
            </>
        );
    };

    private renderMobile = (): ReactNode => {
        const {
            hotelReviews: {isLoadingList},
        } = this.props;

        return (
            <>
                {this.renderReviewTotalCount()}
                {this.renderSortingOptions()}
                {this.renderKeyPhrases()}
                {isLoadingList
                    ? this.renderSkeleton(true)
                    : this.renderTextReviews()}
            </>
        );
    };

    private renderContent(): ReactNode {
        const {
            deviceType: {isMobile},
        } = this.props;

        return isMobile ? this.renderMobile() : this.renderDesktop();
    }

    render(): ReactNode {
        const {
            className,
            hotelReviews,
            deviceType: {isMobile},
            isActiveTab,
        } = this.props;
        const {
            isLoading,
            data: {
                textReviews: {length},
            },
        } = hotelReviews;

        return (
            <div
                className={cx(
                    {root_mobile: isMobile, root_isActiveTab: isActiveTab},
                    className,
                )}
                {...prepareQaAttributes('hotelReviews')}
            >
                {isLoading && length === 0
                    ? this.renderSkeleton(false)
                    : this.renderContent()}
            </div>
        );
    }
}

export default HotelReviews;
