import React, {createRef, PureComponent, ReactNode, RefObject} from 'react';

import {
    REVIEWS_LOAD_COUNT,
    REVIEWS_SORTING_INFO,
    REVIEW_PLACEHOLDERS_COUNT,
    SET_REVIEW_REACTION_DEBOUNCE,
} from './constants';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IKeyPhrase, ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {IHotelReviewReactionPayload} from 'reducers/depreacted/hotels/hotelPage/reviews/list/types';
import {TSortId} from 'types/hotels/search/ISortInfo';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import debounceById from 'utilities/functions/debounceById';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import scrollContainerToNode from 'utilities/dom/scrollContainerToNode';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';

import * as i18nBlock from 'i18n/hotels-HotelReviews';
import * as i18nCommonBlock from 'i18n/hotels-Common';

import Card from 'components/Card/Card';
import LinkButton from 'components/LinkButton/LinkButton';
import Separator from 'components/Separator/Separator';
import Spinner from 'components/Spinner/Spinner';
import LazyLoad from 'projects/depreacted/hotels/components/LazyLoad/LazyLoad';
import TextHotelReviewsSkeleton from './components/TextHotelReviewsSkeleton/TextHotelReviewsSkeleton';
import HotelReviewWithReaction from './components/TextHotelReview/HotelReviewWithReaction';
import KeyPhraseCheckButton from './components/KeyPhraseCheckButton/KeyPhraseCheckButton';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import HotelsSort from 'projects/depreacted/hotels/components/HotelsSort/HotelsSort';
import HorizontalScroller from 'components/HorizontalScroller/HorizontalScroller';
import Flex from 'components/Flex/Flex';
import UserReview from './components/UserReview/UserReview';
import Rating from 'components/Rating/Rating';
import Anchor from 'components/Anchor/Anchor';

import {IHotelReviewsContainer} from './HotelReviewsContainerRedesign';

/* init Styles */
import cx from './HotelReviewsRedesign.scss';

/* Component Types */
interface IHotelReviewsProps
    extends IHotelReviewsContainer,
        IWithClassName,
        IWithDeviceType {
    isActiveTab: boolean;
    isLazyLoading?: boolean;
    maxReviewRenderCount?: number;
    reviewsRef?: React.RefObject<HTMLDivElement>;

    onMoreReviews?: () => void;
    onKeyPhraseClick?: (keyPhrase: string) => void;
}

class HotelReviewsRedesign extends PureComponent<IHotelReviewsProps> {
    /* Helpers */

    private debouncedReactionRequest = debounceById(
        (reactionPayload: IHotelReviewReactionPayload) => {
            const {setHotelReviewReactionRequest} = this.props;

            setHotelReviewReactionRequest(reactionPayload);
        },
        SET_REVIEW_REACTION_DEBOUNCE,
    );

    keyPhrasesRef: RefObject<HTMLDivElement> = createRef();

    /* Handlers */

    private handleKeyPhraseClick = (
        keyPhraseEvent: string,
        element: HTMLElement,
    ): void => {
        const {
            filterHotelReviews,
            hotelReviews: {activeKeyPhrase},
            deviceType,
        } = this.props;

        const keysPhrase =
            activeKeyPhrase === keyPhraseEvent ? undefined : keyPhraseEvent;

        if (deviceType.isMobile && this.keyPhrasesRef.current) {
            scrollContainerToNode(this.keyPhrasesRef.current, element, {
                horizontal: true,
                horizontalPadding: 16,
            });
        }

        filterHotelReviews({
            textReviewLimit: REVIEWS_LOAD_COUNT,
            keyPhraseFilter: keysPhrase,
        });

        if (keysPhrase) {
            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_TAG_CLICK, {
                hotels: {activeKeyPhrase: keysPhrase},
            });
        }
    };

    private handleReviewsButtonClick = (): void => {
        const {
            hotelReviews: {
                data,
                isLoading,
                activeKeyPhrase,
                currentSortingOption,
            },
            getHotelReviews,
        } = this.props;

        if (!isLoading && data.textReviews.length < data.totalTextReviewCount) {
            getHotelReviews({
                textReviewOffset: data.textReviews.length,
                textReviewLimit: REVIEWS_LOAD_COUNT,
                keyPhraseFilter: activeKeyPhrase,
                textReviewRanking: currentSortingOption,
            });
            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_MORE_CLICK, {
                hotels: {
                    moreReviewsOffset: data.textReviews.length,
                    activeKeyPhrase,
                },
            });
        }
    };

    private handleSetReviewReaction = (
        reactionPayload: IHotelReviewReactionPayload,
    ): void => {
        const {setHotelReviewReaction} = this.props;

        setHotelReviewReaction(reactionPayload);
        this.debouncedReactionRequest(
            reactionPayload.reviewId,
            reactionPayload,
        );
    };

    private handleReviewsSortingOptionsChange = ({id}: {id: TSortId}): void => {
        const {
            hotelReviews: {isLoading, activeKeyPhrase},
            sortingHotelReviews,
        } = this.props;

        if (!isLoading) {
            sortingHotelReviews({
                textReviewOffset: 0,
                textReviewLimit: REVIEWS_LOAD_COUNT,
                keyPhraseFilter: activeKeyPhrase,
                textReviewRanking: id,
            });
        }
    };

    /* Render */

    private renderAnchor(): ReactNode {
        return (
            <Anchor
                anchorIds={[
                    'HOTEL_REVIEWS_ANCHOR',
                    'HOTEL_CREATE_REVIEW_ANCHOR',
                ]}
            />
        );
    }

    private renderReviewTotalCount(): ReactNode {
        const {hotelReviews, deviceType, rating} = this.props;

        const {
            data: {totalTextReviewCount},
        } = hotelReviews;

        return (
            <Heading
                level={deviceType.isDesktop ? 2 : 3}
                className={cx(
                    'totalReviewCount',
                    deviceModMobile('totalReviewCount', deviceType),
                )}
                {...prepareQaAttributes('hotelPageReviewsTitle')}
            >
                <Rating className={cx('rating')} size="l" rating={rating} />
                {i18nCommonBlock.totalReviewCount({
                    totalReviewCount: totalTextReviewCount,
                })}
            </Heading>
        );
    }

    private renderTextReview = (textReview: ITextReview): ReactNode => {
        const {
            userInfo,
            deviceType: {isMobile},
        } = this.props;

        const textHotelReviewNode = (
            <HotelReviewWithReaction
                className={cx('textReview')}
                key={textReview.id}
                isAuth={isAuthUser(userInfo)}
                review={textReview}
                onHotelReviewReaction={this.handleSetReviewReaction}
            />
        );

        return isMobile ? (
            <Card
                className={cx('textReviewWrapper')}
                shadow="default"
                key={textReview.id}
            >
                {textHotelReviewNode}
            </Card>
        ) : (
            textHotelReviewNode
        );
    };

    private renderLoader(): ReactNode {
        const {deviceType} = this.props;

        return (
            <Spinner
                className={cx('loader')}
                size={deviceType.isMobile ? 's' : 'xxs'}
                {...prepareQaAttributes('moreReviewsLoader')}
            />
        );
    }

    private renderTextReviews(): ReactNode {
        const {
            hotelReviews,
            isLazyLoading,
            onMoreReviews,
            maxReviewRenderCount,
        } = this.props;

        const {
            isLoading,
            isLoadingList,
            data: {textReviews},
        } = hotelReviews;

        const resultTextReviews = maxReviewRenderCount
            ? textReviews.slice(0, maxReviewRenderCount)
            : textReviews;

        return (
            <div className={cx('reviewsWrap')}>
                {isLazyLoading && onMoreReviews ? (
                    <LazyLoad
                        isLoading={isLoading || isLoadingList}
                        placeholdersCount={REVIEW_PLACEHOLDERS_COUNT}
                        placeholderNode={this.renderLoader()}
                        onBottomScroll={onMoreReviews}
                    >
                        {resultTextReviews.map(this.renderTextReview)}
                    </LazyLoad>
                ) : (
                    resultTextReviews.map(this.renderTextReview)
                )}
            </div>
        );
    }

    private renderSortingOptions(): ReactNode {
        const {
            deviceType,
            hotelReviews: {currentSortingOption, data},
        } = this.props;

        if (!data?.totalTextReviewCount) return null;

        const sortInfo = {
            ...REVIEWS_SORTING_INFO,
            selectedSortId: currentSortingOption,
        };

        return (
            <HotelsSort
                sortInfo={sortInfo}
                deviceType={deviceType}
                needSyncSortWithServer={false}
                setActiveSort={this.handleReviewsSortingOptionsChange}
                size="s"
                showIcon={deviceType.isMobile}
                className={cx('hotelsSort')}
                {...prepareQaAttributes('reviewsSortBar')}
            />
        );
    }

    private renderKeyPhrases(): ReactNode {
        const {
            hotelReviews: {
                data: {keyPhrases},
            },
            deviceType: {isMobile},
        } = this.props;

        if (isMobile) {
            return (
                <HorizontalScroller scrollerRef={this.keyPhrasesRef}>
                    <Flex
                        inline
                        nowrap
                        between={3}
                        className={cx('keyPhrases')}
                        {...prepareQaAttributes('keyPhrases')}
                    >
                        {keyPhrases.map(this.renderKeyPhrase)}
                    </Flex>
                </HorizontalScroller>
            );
        }

        return (
            <div
                className={cx('keyPhrases')}
                {...prepareQaAttributes('keyPhrases')}
            >
                {keyPhrases.map(this.renderKeyPhrase)}
            </div>
        );
    }

    renderKeyPhrase = (keyPhrase: IKeyPhrase): ReactNode => {
        const {
            hotelReviews: {activeKeyPhrase},
        } = this.props;

        return (
            <KeyPhraseCheckButton
                className={cx('keyPhrase')}
                keyPhrase={keyPhrase}
                key={keyPhrase.name}
                checked={keyPhrase.name === activeKeyPhrase}
                onClick={this.handleKeyPhraseClick}
            />
        );
    };

    private renderMoreReviewsButton(): ReactNode {
        const {
            hotelReviews,
            deviceType: {isMobile, isDesktop},
        } = this.props;
        const {data, isLoading} = hotelReviews;

        const moreReviewCount =
            data.totalTextReviewCount - data.textReviews.length;

        if (moreReviewCount > 0) {
            return (
                <div className={cx('moreReviewsButton')}>
                    {isDesktop && (
                        <>
                            <LinkButton
                                disabled={isLoading}
                                onClick={this.handleReviewsButtonClick}
                                {...prepareQaAttributes('moreReviewsButton')}
                            >
                                {i18nBlock.moreReviews()}
                            </LinkButton>
                            {isLoading && this.renderLoader()}
                        </>
                    )}
                    {isMobile && (
                        <>
                            {isLoading ? (
                                <Flex
                                    justifyContent="center"
                                    alignItems="center"
                                    className={cx('loaderContainer')}
                                >
                                    {this.renderLoader()}
                                </Flex>
                            ) : (
                                <Button
                                    width="max"
                                    size="l"
                                    onClick={this.handleReviewsButtonClick}
                                    {...prepareQaAttributes(
                                        'moreReviewsButton',
                                    )}
                                >
                                    {i18nBlock.moreReviews()}
                                </Button>
                            )}
                        </>
                    )}
                </div>
            );
        }

        return null;
    }

    private renderSkeleton(): ReactNode {
        const {deviceType} = this.props;

        return (
            <TextHotelReviewsSkeleton
                deviceType={deviceType}
                {...prepareQaAttributes('hotelReviewsSkeleton')}
            />
        );
    }

    private renderDesktop = (): ReactNode => {
        const {
            isLazyLoading,
            hotelReviews: {isLoadingList, data},
        } = this.props;
        const {userTextReview} = data || {};

        return (
            <>
                {this.renderAnchor()}
                {this.renderReviewTotalCount()}
                {this.renderSortingOptions()}
                {this.renderKeyPhrases()}
                <Separator />
                <UserReview
                    className={cx('createReview')}
                    userReview={userTextReview}
                />
                <Separator className={cx('createReviewSeparator')} />
                {isLoadingList
                    ? this.renderSkeleton()
                    : this.renderTextReviews()}
                {!isLazyLoading &&
                    !isLoadingList &&
                    this.renderMoreReviewsButton()}
            </>
        );
    };

    private renderMobile = (): ReactNode => {
        const {
            hotelReviews: {isLoadingList, data},
        } = this.props;
        const {userTextReview} = data || {};

        return (
            <>
                {this.renderAnchor()}
                {this.renderReviewTotalCount()}
                {this.renderSortingOptions()}
                {this.renderKeyPhrases()}
                <Card shadow="default" x={4} y={4}>
                    <UserReview userReview={userTextReview} />
                </Card>
                {isLoadingList
                    ? this.renderSkeleton()
                    : this.renderTextReviews()}
                {!isLoadingList && this.renderMoreReviewsButton()}
            </>
        );
    };

    private renderContent(): ReactNode {
        const {
            deviceType: {isMobile},
        } = this.props;

        return isMobile ? this.renderMobile() : this.renderDesktop();
    }

    render(): ReactNode {
        const {className, deviceType, isActiveTab, reviewsRef} = this.props;

        return (
            <div
                ref={reviewsRef}
                className={cx(
                    'root',
                    deviceMods('root', deviceType),
                    {root_isActiveTab: isActiveTab},
                    className,
                )}
                {...prepareQaAttributes('hotelReviews')}
            >
                {this.renderContent()}
            </div>
        );
    }
}

export default HotelReviewsRedesign;
