import React from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';

import {getHotelFeatureRatings} from 'selectors/depreacted/hotels/hotel/reviewsTab/getHotelFeatureRatings';

import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import {CardWithDeviceLayout} from 'components/CardWithDeviceLayout/CardWithDeviceLayout';

import cx from './HotelFeatures.scss';

const getRatingMod = (percent: number) => {
    if (percent >= 80) {
        return 'great';
    }

    if (percent >= 60) {
        return 'fine';
    }

    if (percent >= 20) {
        return 'good';
    }

    return 'none';
};

export interface IHotelFeaturesProps extends IWithClassName {}

const HotelFeatures: React.FC<IHotelFeaturesProps> = ({className}) => {
    const deviceType = useDeviceType();
    const featureRatings = useSelector(getHotelFeatureRatings);

    const lines = featureRatings.map(({id, name, positivePercent}) => {
        const fillStyle = {width: `${positivePercent}%`};
        const borderStyle = {
            width: `calc(${positivePercent}% + 2px)`,
        };

        return (
            <div
                key={id}
                className={cx('feature', deviceMods('feature', deviceType))}
            >
                <div>
                    <span className={cx('feature__text')}>{name}</span>
                    <span className={cx('feature__value')}>
                        {`${positivePercent}%`}
                    </span>
                </div>
                <div className={cx('line')}>
                    <div
                        className={cx(
                            'line__left',
                            `line__left_${getRatingMod(positivePercent)}`,
                        )}
                        style={fillStyle}
                    />
                    <div className={cx('line__border')} style={borderStyle} />
                </div>
            </div>
        );
    });
    const nodeLines = deviceType.isMobile ? (
        <CardWithDeviceLayout>{lines}</CardWithDeviceLayout>
    ) : (
        lines
    );

    const canRenderFeatures = featureRatings && featureRatings.length > 0;

    if (!canRenderFeatures) {
        return null;
    }

    return (
        <div className={className} {...prepareQaAttributes('hotelFeatures')}>
            <div className={cx('title', deviceModMobile('title', deviceType))}>
                {i18nBlock.guestsLiked()}
            </div>
            {nodeLines}
        </div>
    );
};

export default HotelFeatures;
