import {FunctionComponent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {ITextReviewAuthor} from 'types/hotels/hotel/IHotelTextReview';
import {EHotelImageYapicSize} from 'types/hotels/common/EHotelImageYapicSize';

import {humanFormatDate} from 'utilities/dateUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {parseYapicImageUrlTemplate} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {useUserInfo} from 'utilities/hooks/useUserInfo';
import getAvatarUrl from 'utilities/avatar/getAvatarUrl';
import {DEFAULT_AVATAR_ID} from 'utilities/avatar/getAvatarId';

import Link from 'components/Link/Link';
import Stars from 'components/Stars/Stars';
import Avatar from 'components/Avatar/Avatar';

/* Init Styles */
import cx from './TextReviewAuthor.scss';

/* Constants */
const MAX_REVIEW_STARS = 5;

/* Component Types */
interface ITextReviewAuthorProps extends IWithClassName {
    author: ITextReviewAuthor;
    rating: number;
    date: string;
}

const TextReviewAuthor: FunctionComponent<ITextReviewAuthorProps> = props => {
    const {
        className,
        rating,
        date,
        author: {avatarUrl, name, profileUrl},
    } = props;

    const userInfo = useUserInfo();

    const defaultAvatarUrl = getAvatarUrl(
        userInfo.avatarPath,
        DEFAULT_AVATAR_ID,
        {retina: true},
    );

    const avatarSrc = avatarUrl
        ? parseYapicImageUrlTemplate(
              avatarUrl,
              EHotelImageYapicSize.RETINA_MIDDLE,
          )
        : defaultAvatarUrl;

    return (
        <section className={cx(className, 'author')}>
            <Link
                target="_blank"
                rel="noopener noreferrer"
                url={profileUrl}
                className={cx('avatar')}
            >
                <Avatar src={avatarSrc} size="m" />
            </Link>
            <div className={cx('authorInfo')}>
                <div className={cx('about')}>
                    <Link
                        target="_blank"
                        rel="noopener noreferrer"
                        url={profileUrl}
                        className={cx('name')}
                        {...prepareQaAttributes('authorName')}
                    >
                        {name}
                    </Link>
                    <span
                        className={cx('date')}
                        {...prepareQaAttributes('hotelReviewDate')}
                    >
                        {humanFormatDate(date)}
                    </span>
                </div>
                <Stars
                    className={cx('stars')}
                    starCount={rating}
                    totalStarCount={MAX_REVIEW_STARS}
                />
            </div>
        </section>
    );
};

export default TextReviewAuthor;
