import {useCallback, ReactNode, FunctionComponent} from 'react';
import {useSelector} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {UserReactionType} from 'types/hotels/hotel/IHotelTextReview';

import {getUserInfo} from 'selectors/common/userInfoSelector';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {EPassportMode, getPassportUrl} from 'utilities/url/getPassportUrl';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import LikeFilledIcon from 'icons/16/LikeFilled';
import LikeOutlineIcon from 'icons/16/LikeOutline';
import DislikeFilledIcon from 'icons/16/DislikeFilled';
import DislikeOutlineIcon from 'icons/16/DislikeOutline';

/* Init Styles */
import cx from './TextReviewLikes.scss';

/* Component Types */
interface ITextReviewLikesProps extends IWithClassName {
    isAuth: boolean;
    totalLikeCount: number;
    totalDislikeCount: number;
    userReaction: UserReactionType;
    onUserReactionClick: (reaction: UserReactionType) => void;
}

const TextReviewLikes: FunctionComponent<ITextReviewLikesProps> = props => {
    const {
        className,
        isAuth,
        userReaction,
        totalLikeCount,
        totalDislikeCount,
        onUserReactionClick,
    } = props;

    const deviceType = useDeviceType();
    const {passportPath} = useSelector(getUserInfo);

    const redirectToAuth = useCallback(() => {
        window.location.href = getPassportUrl({
            mode: EPassportMode.AUTH,
            passportHost: passportPath,
            retpath: window.location.href,
        });
    }, [passportPath]);

    const handleLikeClick = useCallback(() => {
        if (isAuth) {
            const reaction =
                userReaction === UserReactionType.LIKE
                    ? UserReactionType.NONE
                    : UserReactionType.LIKE;

            onUserReactionClick(reaction);
        } else {
            redirectToAuth();
        }
    }, [isAuth, onUserReactionClick, userReaction, redirectToAuth]);

    const handleDislikeClick = useCallback(() => {
        if (isAuth) {
            const reaction =
                userReaction === UserReactionType.DISLIKE
                    ? UserReactionType.NONE
                    : UserReactionType.DISLIKE;

            onUserReactionClick(reaction);
        } else {
            redirectToAuth();
        }
    }, [isAuth, onUserReactionClick, userReaction, redirectToAuth]);

    const renderLike = (): ReactNode => {
        const isActive = userReaction === UserReactionType.LIKE;
        const isUnselected = userReaction === UserReactionType.DISLIKE;

        return (
            <span
                onClick={handleLikeClick}
                className={cx('linkButton', {
                    linkButton_hover: deviceType.isDesktop,
                    linkButton_like: isActive,
                    linkButton_unselected: isUnselected,
                })}
                {...prepareQaAttributes('hotelReviewLikeButton')}
            >
                {isActive ? (
                    <LikeFilledIcon className={cx('icon')} />
                ) : (
                    <LikeOutlineIcon className={cx('icon')} />
                )}
                <span className={cx('total')}>
                    {i18nBlock.likesCount({count: totalLikeCount})}
                </span>
            </span>
        );
    };

    const renderDislike = (): ReactNode => {
        const isActive = userReaction === UserReactionType.DISLIKE;
        const isUnselected = userReaction === UserReactionType.LIKE;

        return (
            <span
                onClick={handleDislikeClick}
                className={cx('linkButton', {
                    linkButton_hover: deviceType.isDesktop,
                    linkButton_dislike: isActive,
                    linkButton_unselected: isUnselected,
                })}
                {...prepareQaAttributes('hotelReviewDislikeButton')}
            >
                {isActive ? (
                    <DislikeFilledIcon className={cx('icon')} />
                ) : (
                    <DislikeOutlineIcon className={cx('icon')} />
                )}
                <span className={cx('total')}>{totalDislikeCount}</span>
            </span>
        );
    };

    return (
        <div className={cx('likes', className)}>
            {renderLike()}
            {renderDislike()}
        </div>
    );
};

export default TextReviewLikes;
