import React from 'react';

import {IWithClassName} from 'types/withClassName';

import capitalizeFirstLetter from 'utilities/strings/capitalizeFirstLetter';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import getAvatarUrl from 'utilities/avatar/getAvatarUrl';
import getAvatarId from 'utilities/avatar/getAvatarId';
import {isAuthUser} from 'utilities/userInfo/isAuthUser';
import {useUserInfo} from 'utilities/hooks/useUserInfo';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import RatingStars from 'components/RatingStars/RatingStars';
import Avatar from 'components/Avatar/Avatar';

import cx from './CreateReviewTrigger.scss';

interface ICreateReviewTriggerProps extends IWithClassName {
    onStarClick: (index: number) => void;
    ref?: React.Ref<HTMLElement>;
}

const CreateReviewTrigger: React.FC<ICreateReviewTriggerProps> =
    React.forwardRef<HTMLElement, ICreateReviewTriggerProps>(
        ({className, onStarClick}, ref) => {
            const userInfo = useUserInfo();
            const deviceType = useDeviceType();
            const {isMobile} = deviceType;

            const avatarId = getAvatarId(userInfo);
            const avatarUrl = getAvatarUrl(userInfo.avatarPath, avatarId, {
                retina: true,
            });

            return (
                <Flex
                    className={cx(deviceMods('root', deviceType), className)}
                    flexDirection={isMobile ? 'column' : 'row'}
                    alignItems={isMobile ? 'flex-start' : 'center'}
                    ref={ref}
                >
                    <Flex flexDirection="row" alignItems="center">
                        <Avatar src={avatarUrl} size="m" />
                        <Flex flexDirection="column" className={cx('userWrap')}>
                            {isAuthUser(userInfo) && (
                                <Text size="m" weight="bold">
                                    {userInfo.displayName}
                                </Text>
                            )}
                            <Text size="m">
                                {getRateAndWriteReviewString(
                                    isAuthUser(userInfo),
                                )}
                            </Text>
                        </Flex>
                    </Flex>
                    <RatingStars
                        className={cx('ratingStars')}
                        onClick={onStarClick}
                    />
                </Flex>
            );
        },
    );

CreateReviewTrigger.displayName = 'CreateReviewTrigger';

function getRateAndWriteReviewString(isAuth: boolean): string {
    if (isAuth) {
        return i18nBlock.rateAndWriteReview();
    }

    return capitalizeFirstLetter(i18nBlock.rateAndWriteReview());
}

export default CreateReviewTrigger;
