import React, {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {ITextReview} from 'types/hotels/hotel/IHotelTextReview';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {deleteHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/delete/actions';

import deleteHotelReview from 'selectors/depreacted/hotels/hotel/reviewsTab/deleteHotelReview';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {useBoolean} from 'utilities/hooks/useBoolean';
import {useTimeout} from 'utilities/hooks/useTimeout';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import Flex from 'components/Flex/Flex';
import Button from 'components/Button/Button';
import Snackbar from 'components/Snackbar/Snackbar';

import cx from './DeleteReviewDialog.scss';

const SNACKBAR_HIDE_TIMEOUT = 3000;

interface IDeleteReviewDialogProps {
    isVisible: boolean;
    userReview?: ITextReview;
    onClose: () => void;
}

const DeleteReviewDialog: React.FC<IDeleteReviewDialogProps> = ({
    isVisible,
    userReview,
    onClose,
}) => {
    const deviceType = useDeviceType();
    const {isMobile, isDesktop} = deviceType;
    const dispatch = useDispatch();
    const {isSubmitting, isSuccess} = useSelector(deleteHotelReview);
    const {
        value: isSnackbarVisible,
        setTrue: setSnackbarVisible,
        setFalse: setSnackbarHidden,
    } = useBoolean(false);
    const setSnackbarHiddenWithTimeout = useTimeout(
        setSnackbarHidden,
        SNACKBAR_HIDE_TIMEOUT,
    );

    const deleteReview = useCallback(() => {
        if (userReview) {
            dispatch(
                deleteHotelReviewActions.request({
                    reviewId: userReview.id,
                }),
            );
            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_DELETE_CLICK);
        }
    }, [dispatch, userReview]);

    useEffect(() => {
        if (isSubmitting) {
            onClose();
        }
    }, [isSubmitting, onClose]);

    useEffect(() => {
        if (isSuccess) {
            setSnackbarVisible();
            setSnackbarHiddenWithTimeout();
        }
    }, [isSuccess, setSnackbarVisible, setSnackbarHiddenWithTimeout]);

    return (
        <>
            <ModalOrBottomSheet
                isVisible={isVisible}
                onClose={onClose}
                modalProps={{hasCloseButton: false}}
                bottomSheetProps={{
                    hasCloseButton: true,
                }}
            >
                <Flex
                    flexDirection="column"
                    alignItems="center"
                    className={cx(deviceMods('root', deviceType))}
                >
                    <Heading
                        level={isDesktop ? '2' : '3'}
                        fontWeight="bold"
                        className={cx('title')}
                    >
                        {i18nBlock.deleteReviewTitle()}
                    </Heading>
                    <Text className={cx('description')}>
                        {i18nBlock.deleteReviewDescription()}
                    </Text>
                    <Flex
                        between={4}
                        inline
                        flexDirection="row"
                        alignItems="center"
                        className={cx('buttons')}
                    >
                        <Button
                            theme="primary"
                            width={isMobile ? 'max' : undefined}
                            className={cx('button')}
                            size="l"
                            onClick={deleteReview}
                        >
                            {i18nBlock._delete()}
                        </Button>
                        {isDesktop && (
                            <Button
                                theme="secondary"
                                size="l"
                                className={cx('button')}
                                onClick={onClose}
                            >
                                {i18nBlock.cancel()}
                            </Button>
                        )}
                    </Flex>
                </Flex>
            </ModalOrBottomSheet>
            <Snackbar isVisible={isSnackbarVisible}>
                {i18nBlock.reviewDeleted()}
            </Snackbar>
        </>
    );
};

export default DeleteReviewDialog;
