import React, {useCallback, useEffect} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {IAddHotelReviewRequest} from 'server/api/HotelSearchAPI/types/IAddHotelReview';
import {IEditHotelReviewRequest} from 'server/api/HotelSearchAPI/types/IEditHotelReview';

import {editHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/edit/actions';
import {addHotelReviewActions} from 'reducers/depreacted/hotels/hotelPage/reviews/add/actions';

import addHotelReview from 'selectors/depreacted/hotels/hotel/reviewsTab/addHotelReview';
import editHotelReview from 'selectors/depreacted/hotels/hotel/reviewsTab/editHotelReview';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-HotelReviews';

import ModalOrBottomSheet from 'components/ModalOrBottomSheet/ModalOrBottomSheet';
import Flex from 'components/Flex/Flex';
import CommonError from 'icons/illustrations/CommonError/CommonErrorM';
import Heading from 'components/Heading/Heading';
import Text from 'components/Text/Text';
import Button from 'components/Button/Button';

import cx from './ErrorDialog.scss';

interface IErrorDialogProps {
    isVisible: boolean;
    retryValues: IAddHotelReviewRequest | IEditHotelReviewRequest;
    onClose: () => void;
}

const ErrorDialog: React.FC<IErrorDialogProps> = ({
    isVisible,
    retryValues,
    onClose,
}) => {
    const deviceType = useDeviceType();
    const addReview = useSelector(addHotelReview);
    const editReview = useSelector(editHotelReview);
    const dispatch = useDispatch();

    const handleRetryClick = useCallback(() => {
        if ('reviewId' in retryValues) {
            dispatch(editHotelReviewActions.request(retryValues));
        } else {
            dispatch(addHotelReviewActions.request(retryValues));
        }
    }, [dispatch, retryValues]);

    useEffect(() => {
        if (addReview.isSuccess || editReview.isSuccess) {
            onClose();
        }
    }, [addReview, editReview, onClose]);

    return (
        <ModalOrBottomSheet
            isVisible={isVisible}
            onClose={onClose}
            bottomSheetProps={{hasCloseButton: true}}
        >
            <Flex
                flexDirection="column"
                alignItems="center"
                className={cx(deviceMods('root', deviceType))}
            >
                <CommonError className={cx('image')} />
                <Heading level="2">{i18nBlock.errorTitle()}</Heading>
                <Text align="center" className={cx('description')}>
                    {i18nBlock.errorDescription()}
                </Text>
                <Button
                    theme="primary"
                    width="max"
                    size="l"
                    className={cx('button')}
                    onClick={handleRetryClick}
                    disabled={addReview.isSubmitting || editReview.isSubmitting}
                >
                    {i18nBlock.errorButton()}
                </Button>
            </Flex>
        </ModalOrBottomSheet>
    );
};

export default ErrorDialog;
