import {useCallback, useContext} from 'react';

import {
    INewImagesSelectorImage,
    TImagesSelectorImage,
} from 'components/ImagesSelector/types/IImagesSelectorImage';
import EAsyncStatus from 'types/common/EAsyncStatus';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {reachGoal} from 'utilities/metrika';

import {hotelSearchService} from 'serviceProvider';

import {ImageUploaderContext} from './ImageUploaderContext';

interface IUseImageUploader {
    images: readonly INewImagesSelectorImage[];
    uploadImages: (images: INewImagesSelectorImage[]) => void;
    deleteImage: (image: TImagesSelectorImage) => void;
    deleteUnsavedImages: () => void;
    deleteServerImages: () => void;
    hasUploadingImages: () => boolean;
}

export const useImageUploader = (hotelSlug?: string): IUseImageUploader => {
    const context = useContext(ImageUploaderContext);

    const deleteImage = useCallback(
        image => {
            if (image.file) {
                if (image.status === EAsyncStatus.SUCCESS) {
                    context.removeImage(image);
                    hotelSearchService
                        .provider()
                        .deleteHotelReviewImages(
                            {hotelSlug},
                            {imageIds: [image.id]},
                        );
                } else {
                    image.shouldBeDeleted = true;
                }
            } else {
                context.deletedServerImageIds.push(image.id);
            }
        },
        [context, hotelSlug],
    );
    const deleteServerImages = useCallback(() => {
        if (context.deletedServerImageIds.length) {
            hotelSearchService.provider().deleteHotelReviewImages(
                {hotelSlug},
                {
                    imageIds: [...context.deletedServerImageIds],
                },
            );
            context.deletedServerImageIds.length = 0;
        }
    }, [context, hotelSlug]);
    const deleteUnsavedImages = useCallback(() => {
        const toDeleteNow = context.images.filter(
            i => i.status === EAsyncStatus.SUCCESS,
        );

        if (toDeleteNow.length) {
            hotelSearchService.provider().deleteHotelReviewImages(
                {hotelSlug},
                {
                    imageIds: context.images.map(({id}) => id).filter(Boolean),
                },
            );
            toDeleteNow.forEach(context.removeImage);
        }

        // если пользователь закрыл окно не дождавшись загрузки файла
        context.images
            .filter(i => i.status === EAsyncStatus.LOADING)
            .forEach(i => {
                i.shouldBeDeleted = true;
            });
    }, [hotelSlug, context]);

    const hasUploadingImages = useCallback(() => {
        return Boolean(
            context.images.some(
                i => 'status' in i && i.status === EAsyncStatus.LOADING,
            ),
        );
    }, [context]);

    const uploadImage = useCallback(
        async (image: INewImagesSelectorImage) => {
            if (!image.file) {
                return;
            }

            const formData = new FormData();

            formData.append('image', image.file);

            image.status = EAsyncStatus.LOADING;
            context.addImage(image);

            try {
                const response = await hotelSearchService
                    .provider()
                    .uploadHotelReviewImage(formData, {hotelSlug});

                image.id = response.data.imageId;
                image.status = EAsyncStatus.SUCCESS;

                // если пользователь закрыл окно не дождавшись загрузки файла
                if (image.shouldBeDeleted) {
                    deleteImage(image);
                }
            } catch (e) {
                image.status = EAsyncStatus.ERROR;
            }

            context.imageStateChange(image);

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_PHOTO_LOAD);
        },
        [hotelSlug, context, deleteImage],
    );

    const uploadImages = useCallback(
        (images: INewImagesSelectorImage[]) => images.forEach(uploadImage),
        [uploadImage],
    );

    return {
        images: context.images,
        uploadImages,
        deleteImage,
        deleteUnsavedImages,
        deleteServerImages,
        hasUploadingImages,
    };
};
