import {ComponentType, createRef, PureComponent, ReactNode} from 'react';

import {
    ERefundType,
    ICancellationInfo,
} from 'types/hotels/offer/IHotelOfferCancellationInfo';
import {IWithClassName} from 'types/withClassName';
import {IIconProps} from 'icons/types/icon';
import {IBookDeferredPaymentSchedule} from 'server/api/HotelsBookAPI/types/IBookOffer';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {IDevice} from 'reducers/common/commonReducerTypes';

import getCancellationLabelOptions from './utilities/getCancellationLabelOptions';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-CancellationInfo';

import Text, {TextColor, TTextWeight} from 'components/Text/Text';
import MessageBoxPopup, {
    EMessageBoxPopupTheme,
} from 'components/MessageBoxPopup/MessageBoxPopup';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import InfoIcon from 'icons/16/Info';
import BottomSheet from 'components/BottomSheet/BottomSheet';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import HotelsCancellationTextInfo from '../HotelsCancellationTextInfo/HotelsCancellationTextInfo';

import cx from './HotelsCancellationInfo.scss';

export enum EFormatLabel {
    FULL_TEXT_WITH_DATE_AND_TIME = 'fullTextWithDateAndTime',
    FULL_TEXT = 'fullText',
    ALWAYS_FULL_TEXT = 'alwaysFullText',
    ONLY_REFUND_TEXT = 'onlyRefundText',
}

interface IHotelsCancellationInfoProps
    extends IWithClassName,
        IWithQaAttributes {
    cancellationInfo: ICancellationInfo;
    labelTextSize?: 's' | 'm';
    labelTextColor?: TextColor;
    labelTextWeight?: TTextWeight;
    labelIconLeft?: ComponentType<IIconProps>;
    labelIconLeftClassName?: string;
    canRenderOnlyFullyRefundable?: boolean;
    formatLabel?: EFormatLabel;
    preventRenderDetailedInfo?: boolean;
    deferredPaymentSchedule?: IBookDeferredPaymentSchedule;

    /**
     * Отображать ли цену в лейбле отмены со штрафом
     */
    isLabelPenaltyWithPrice?: boolean;
    textWithIconClassName?: string;
    shortDate?: boolean;
}

interface IHotelsCancellationInfoState {
    isVisibleDetailedCancellationInfo: boolean;
    canRenderClientSizeBottomSheet: boolean;
}

const DIRECTION = [
    EPopupDirection.BOTTOM_RIGHT,
    EPopupDirection.TOP_RIGHT,
    EPopupDirection.BOTTOM_LEFT,
];
const POPUP_SECONDARY_OFFSET = -20;
const POPUP_TAIL_OFFSET = 20;

class HotelsCancellationInfo extends PureComponent<
    IHotelsCancellationInfoProps,
    IHotelsCancellationInfoState
> {
    labelIconRef = createRef<HTMLSpanElement>();

    state = {
        isVisibleDetailedCancellationInfo: false,
        canRenderClientSizeBottomSheet: false,
    };

    componentDidMount(): void {
        this.setState({canRenderClientSizeBottomSheet: true});
    }

    /* Handlers */

    handleCloseMessageBoxPopup = (): void => {
        this.setState({isVisibleDetailedCancellationInfo: false});
    };

    handleLabelClick = (): void => {
        this.setState({
            isVisibleDetailedCancellationInfo: true,
        });
    };

    handleCloseBottomSheet = (): void => {
        this.setState({isVisibleDetailedCancellationInfo: false});
    };

    /* Helpers */

    checkNeedDetailedCancellationInfoRender(): boolean {
        const {
            cancellationInfo: {refundRules},
            preventRenderDetailedInfo,
        } = this.props;

        if (preventRenderDetailedInfo) {
            return false;
        }

        if (refundRules?.length) {
            if (refundRules.length === 1) {
                const [firstRule] = refundRules;

                return (
                    Boolean(firstRule?.startsAt) || Boolean(firstRule?.endsAt)
                );
            }

            return true;
        }

        return false;
    }

    /* Render */

    renderLabel(deviceType: IDevice): ReactNode {
        const {
            className,
            cancellationInfo,
            labelTextColor,
            labelIconLeft,
            labelIconLeftClassName,
            labelTextSize = 'm',
            formatLabel,
            deferredPaymentSchedule,
            isLabelPenaltyWithPrice,
            textWithIconClassName,
            shortDate,
            labelTextWeight,
        } = this.props;
        const {text, color} = getCancellationLabelOptions({
            cancellationInfo,
            formatLabel,
            deferredPaymentSchedule,
            isLabelPenaltyWithPrice,
            shortDate,
        });
        const canDetailedCancellationInfoRender =
            this.checkNeedDetailedCancellationInfoRender();

        return (
            <Text
                color={labelTextColor || color}
                size={labelTextSize}
                weight={labelTextWeight}
                className={cx(className, 'labelRoot')}
                tag="div"
            >
                <TextWithIcon
                    className={cx(
                        'label',
                        deviceModMobile('label', deviceType),
                        {
                            label_type_trigger:
                                canDetailedCancellationInfoRender,
                        },
                        textWithIconClassName,
                    )}
                    text={text}
                    size={labelTextSize}
                    onClick={
                        canDetailedCancellationInfoRender
                            ? this.handleLabelClick
                            : undefined
                    }
                    iconLeft={labelIconLeft}
                    iconLeftClassName={labelIconLeftClassName}
                    iconRightRef={this.labelIconRef}
                    iconRight={
                        canDetailedCancellationInfoRender ? InfoIcon : undefined
                    }
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: `trigger`,
                    })}
                />
            </Text>
        );
    }

    renderCancellationInfo(): ReactNode {
        const {
            cancellationInfo: {refundRules},
        } = this.props;

        if (!refundRules) {
            return null;
        }

        return <HotelsCancellationTextInfo refundRules={refundRules} />;
    }

    renderMessageBoxPopup(): ReactNode {
        const canDetailedCancellationInfoRender =
            this.checkNeedDetailedCancellationInfoRender();

        if (!canDetailedCancellationInfoRender) {
            return null;
        }

        return (
            <MessageBoxPopup
                anchorRef={this.labelIconRef}
                isVisible={this.state.isVisibleDetailedCancellationInfo}
                onClose={this.handleCloseMessageBoxPopup}
                direction={DIRECTION}
                secondaryOffset={POPUP_SECONDARY_OFFSET}
                tailOffset={POPUP_TAIL_OFFSET}
                theme={EMessageBoxPopupTheme.WHITE}
            >
                <div className={cx('messageBoxPopupContent')}>
                    {this.renderCancellationInfo()}
                </div>
            </MessageBoxPopup>
        );
    }

    renderBottomSheet(): ReactNode {
        const {canRenderClientSizeBottomSheet} = this.state;
        const canDetailedCancellationInfoRender =
            this.checkNeedDetailedCancellationInfoRender();

        if (!canDetailedCancellationInfoRender) {
            return null;
        }

        if (!canRenderClientSizeBottomSheet) {
            return null;
        }

        return (
            <BottomSheet
                isOpened={this.state.isVisibleDetailedCancellationInfo}
                onClose={this.handleCloseBottomSheet}
            >
                <div className={cx('bottomSheetContent')}>
                    <Text
                        className={cx('bottomSheetContentTitle')}
                        size="m"
                        tag="div"
                        weight="bold"
                        color="primary"
                    >
                        {i18nBlock.mobileCancellationInfoTitle()}
                    </Text>
                    {this.renderCancellationInfo()}
                </div>
            </BottomSheet>
        );
    }

    render(): ReactNode {
        const {
            canRenderOnlyFullyRefundable,
            cancellationInfo: {refundType},
        } = this.props;

        if (
            canRenderOnlyFullyRefundable &&
            refundType !== ERefundType.FULLY_REFUNDABLE
        ) {
            return null;
        }

        return (
            <DeviceTypeContext.Consumer>
                {(deviceType): ReactNode => (
                    <>
                        {this.renderLabel(deviceType)}
                        {deviceType.isMobile
                            ? this.renderBottomSheet()
                            : this.renderMessageBoxPopup()}
                    </>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

export default HotelsCancellationInfo;
