import React from 'react';

import {
    ERefundType,
    ICancellationInfo,
    IRefundRule,
} from 'types/hotels/offer/IHotelOfferCancellationInfo';
import {IBookDeferredPaymentSchedule} from 'server/api/HotelsBookAPI/types/IBookOffer';

import {
    HUMAN,
    HUMAN_DATETIME_WITHOUT_YEAR,
    HUMAN_SHORT,
    HUMAN_SHORT_DATETIME_WITHOUT_YEAR,
} from 'utilities/dateUtils/formats';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/hotels-CancellationInfo';

import {TextColor} from 'components/Text/Text';
import Price from 'components/Price/Price';

import {EFormatLabel} from '../HotelsCancellationInfo';
import {
    checkCancellationDateTimeDiffByNowDate,
    prepareCancellationDateTime,
} from './prepareCancellationDateTime';

export interface ICancellationLabelOptions {
    text: React.ReactNode;
    color: TextColor;
}

const checkRenderFullTextLabel = ({
    refundRule,
    formatLabel,
}: {
    refundRule?: IRefundRule;
    formatLabel: EFormatLabel;
}): boolean => {
    if (!refundRule) {
        return false;
    }

    switch (formatLabel) {
        case EFormatLabel.FULL_TEXT: {
            return checkCancellationDateTimeDiffByNowDate(refundRule.endsAt);
        }

        case EFormatLabel.ALWAYS_FULL_TEXT:
        case EFormatLabel.FULL_TEXT_WITH_DATE_AND_TIME: {
            return Boolean(refundRule.endsAt);
        }

        default: {
            return false;
        }
    }
};

const getCancellationLabelOptions = ({
    cancellationInfo,
    formatLabel = EFormatLabel.FULL_TEXT,
    deferredPaymentSchedule,
    isLabelPenaltyWithPrice,
    shortDate,
}: {
    cancellationInfo: ICancellationInfo;
    formatLabel?: EFormatLabel;
    deferredPaymentSchedule?: IBookDeferredPaymentSchedule;
    isLabelPenaltyWithPrice?: boolean;
    shortDate?: boolean;
}): ICancellationLabelOptions => {
    const {refundType, refundRules} = cancellationInfo;

    switch (refundType) {
        case ERefundType.FULLY_REFUNDABLE: {
            const firstRefundRule = refundRules?.[0];
            const canRenderFullTextLabel = checkRenderFullTextLabel({
                formatLabel,
                refundRule: firstRefundRule,
            });

            if (firstRefundRule) {
                const {endsAt} = firstRefundRule;

                if (formatLabel === EFormatLabel.FULL_TEXT_WITH_DATE_AND_TIME) {
                    return {
                        text: i18nBlock.fullyRefundableWithDateTimeLabelWithNoPenaltyText(
                            {
                                dateTime: prepareCancellationDateTime(
                                    endsAt,
                                    shortDate
                                        ? HUMAN_SHORT_DATETIME_WITHOUT_YEAR
                                        : HUMAN_DATETIME_WITHOUT_YEAR,
                                ),
                            },
                        ),
                        color: 'success',
                    };
                }

                if (
                    firstRefundRule.type === ERefundType.FULLY_REFUNDABLE &&
                    canRenderFullTextLabel
                ) {
                    return {
                        text: i18nBlock.fullyRefundableWithDateTimeLabel({
                            dateTime: prepareCancellationDateTime(
                                endsAt,
                                shortDate ? HUMAN_SHORT : HUMAN,
                            ),
                        }),
                        color: 'success',
                    };
                }
            }

            return {
                text: i18nBlock.fullyRefundableLabel(),
                color: 'success',
            };
        }

        case ERefundType.REFUNDABLE_WITH_PENALTY: {
            if (
                deferredPaymentSchedule &&
                formatLabel === EFormatLabel.FULL_TEXT_WITH_DATE_AND_TIME
            ) {
                return {
                    text: i18nBlock.nonRefundablePrepaymentLabel(),
                    color: 'primary',
                };
            }

            const penaltyAmount = refundRules?.[0]?.penalty;

            if (isLabelPenaltyWithPrice && penaltyAmount) {
                return {
                    text: insertJSXIntoKey(
                        i18nBlock.refundableWithPenaltyWithPriceLabel,
                    )({
                        price: <Price {...penaltyAmount} />,
                    }),
                    color: 'primary',
                };
            }

            return {
                text: i18nBlock.refundableWithPenaltyLabel(),
                color: 'primary',
            };
        }

        case ERefundType.NON_REFUNDABLE: {
            if (
                deferredPaymentSchedule &&
                formatLabel === EFormatLabel.FULL_TEXT_WITH_DATE_AND_TIME
            ) {
                return {
                    text: i18nBlock.nonRefundablePrepaymentLabel(),
                    color: 'primary',
                };
            }

            return {
                text: i18nBlock.nonRefundableLabel(),
                color: 'primary',
            };
        }

        default: {
            return {
                text: i18nBlock.nonRefundableLabel(),
                color: 'primary',
            };
        }
    }
};

export default getCancellationLabelOptions;
