import {PureComponent, ReactNode} from 'react';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IWithClassName} from 'types/withClassName';
import {ISortInfo, TSortId} from 'types/hotels/search/ISortInfo';
import {IWithDeviceType} from 'types/withDeviceType';

import {reachGoal} from 'utilities/metrika';
import browserGeolocation from 'utilities/browserGeolocation/browserGeolocation';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import HotelsSortBar from './components/HotelsSortBar/HotelsSortBar';
import HotelsSortSelect from './components/HotelsSortSelect/HotelsSortSelect';
import SortBottomSheet from './components/SortBottomSheet/SortBottomSheet';

/* Init styles */

import cx from './HotelsSort.scss';

/* Component props */
export interface IHotelsSortProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    sortInfo?: ISortInfo;
    needSyncSortWithServer: boolean;
    setActiveSort: (params: {id: TSortId; sortOrigin?: string}) => void;
    openHotelsGeolocationErrorModal?: () => void;
    size?: 'm' | 's';

    // Если передан true: нарисует иконку в зависимости от значения hint
    showIcon?: boolean;
    sortGroupClassName?: string;
    hotelsHeaderRedesign?: boolean;
}

class HotelsSort extends PureComponent<IHotelsSortProps> {
    private handleSetActiveSort = async (params: {
        id: TSortId;
        requiresGeoLocation?: boolean;
    }): Promise<void> => {
        const {setActiveSort, openHotelsGeolocationErrorModal} = this.props;

        if (params.requiresGeoLocation) {
            const data = await browserGeolocation.getGeolocation();

            if (!data?.sortOrigin) {
                openHotelsGeolocationErrorModal?.();

                return;
            }

            setActiveSort({...data, id: params.id});

            reachGoal(EHotelsGoal.SEARCH_PAGE_SORT_CHANGE);

            return;
        }

        setActiveSort(params);

        reachGoal(EHotelsGoal.SEARCH_PAGE_SORT_CHANGE);
    };

    private renderSearchHotelsSortBar(): ReactNode {
        const {
            sortInfo,
            needSyncSortWithServer,
            size,
            showIcon,
            deviceType,
            className,
        } = this.props;
        const totalAvailableSorts = sortInfo?.availableSortTypeGroups?.length;
        const sortBarClassName = cx('searchHotelsSortBar', className);

        if (sortInfo && totalAvailableSorts) {
            return (
                <HotelsSortBar
                    className={sortBarClassName}
                    sortInfo={sortInfo}
                    needSyncSortWithServer={needSyncSortWithServer}
                    onSetActiveSort={this.handleSetActiveSort}
                    size={size}
                    showIcon={showIcon}
                    deviceType={deviceType}
                    sortGroupClassName={cx('sortGroup')}
                    {...prepareQaAttributes(this.props)}
                />
            );
        }

        return (
            <div className={sortBarClassName}>
                <div className={cx('searchHotelsSortBarSkeleton')} />
            </div>
        );
    }

    private renderSearchHotelsSortSelect(): ReactNode {
        const {
            sortInfo,
            needSyncSortWithServer,
            showIcon,
            deviceType,
            className,
        } = this.props;
        const totalAvailableSorts = sortInfo?.availableSortTypeGroups?.length;

        if (sortInfo && totalAvailableSorts && !needSyncSortWithServer) {
            return (
                <HotelsSortSelect
                    sortInfo={sortInfo}
                    onSetActiveSort={this.handleSetActiveSort}
                    showIcon={showIcon}
                    deviceType={deviceType}
                    className={className}
                    {...prepareQaAttributes(this.props)}
                />
            );
        }

        return <div className={cx('searchHotelsSortSelectSkeleton')} />;
    }

    renderSortBottomSheet(): ReactNode {
        const {sortInfo} = this.props;

        return (
            <SortBottomSheet
                sortInfo={sortInfo}
                onSetActiveSort={this.handleSetActiveSort}
            />
        );
    }

    render(): ReactNode {
        const {
            deviceType: {isDesktop},
            hotelsHeaderRedesign,
        } = this.props;

        if (isDesktop) {
            return this.renderSearchHotelsSortBar();
        }

        if (hotelsHeaderRedesign) {
            return this.renderSortBottomSheet();
        }

        return this.renderSearchHotelsSortSelect();
    }
}

export default HotelsSort;
