import {PureComponent, ReactNode} from 'react';
import _findIndex from 'lodash/findIndex';
import _throttle from 'lodash/throttle';

import {IWithClassName} from 'types/withClassName';
import {
    ISortInfo,
    ISortType,
    ISortTypeGroup,
    TSortId,
} from 'types/hotels/search/ISortInfo';
import {IWithDeviceType} from 'types/withDeviceType';
import {TSetActiveSortFunc} from '../../types/TSetActiveSortFunc';

import getSortGroupIcon from '../../utilities/getSortGroupIcon';
import getSelectedSortType from '../../utilities/getSelectedSortType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */

import LinkButton from 'components/LinkButton/LinkButton';

/* Init styles */
import cx from './HotelsSortBar.scss';

/* Constants */
const SET_SORT_THROTTLE_TIME = 1000;

/* Component Props */
export interface ISearchHotelsSortBarProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    sortInfo: ISortInfo;
    needSyncSortWithServer: boolean;
    onSetActiveSort: TSetActiveSortFunc;
    size?: 'm' | 's';
    showIcon?: boolean;
    sortGroupClassName?: string;
}

class SearchHotelsSortBar extends PureComponent<ISearchHotelsSortBarProps> {
    /* Handlers */

    private handleClickSortItem = _throttle(
        (sortId: TSortId, requiresGeoLocation: boolean): void => {
            const {
                sortInfo: {selectedSortId},
                onSetActiveSort,
                needSyncSortWithServer,
            } = this.props;

            if (selectedSortId !== sortId && !needSyncSortWithServer) {
                onSetActiveSort({id: sortId, requiresGeoLocation});
            }
        },
        SET_SORT_THROTTLE_TIME,
    );

    /* Helpers */

    private getNextSortTypeId = (sortTypes: ISortType[]): TSortId => {
        const {
            sortInfo: {selectedSortId},
        } = this.props;

        const [firstSortType] = sortTypes;
        const selectedSortType = getSelectedSortType(sortTypes, selectedSortId);

        if (selectedSortType) {
            const selectedSortTypeIndex = _findIndex(
                sortTypes,
                selectedSortType,
            );
            const nextSortTypeIndex =
                (selectedSortTypeIndex + 1) % sortTypes.length;

            return sortTypes[nextSortTypeIndex].id;
        }

        return firstSortType.id;
    };

    /* Render */

    private renderSortGroup = (sortItem: ISortTypeGroup): ReactNode => {
        const {
            sortInfo: {selectedSortId},
            showIcon,
            deviceType,
            sortGroupClassName,
        } = this.props;

        const {name, sortTypes, requiresGeoLocation} = sortItem;
        const selectedSortType = getSelectedSortType(sortTypes, selectedSortId);
        const isActiveGroup = Boolean(selectedSortType);
        const SortGroupIcon = getSortGroupIcon(
            deviceType,
            sortTypes,
            selectedSortId,
        );
        const canPreventSelectGroup = isActiveGroup && !SortGroupIcon;
        const nextSortTypeId = this.getNextSortTypeId(sortTypes);

        return (
            <LinkButton
                key={name}
                className={cx('sortBarGroup', sortGroupClassName, {
                    sortBarGroup_canPreventSelect: canPreventSelectGroup,
                })}
                onClick={this.handleClickSortItem.bind(
                    this,
                    nextSortTypeId,
                    requiresGeoLocation,
                )}
                theme={isActiveGroup ? 'black' : 'normal'}
            >
                <span>{name}</span>
                {showIcon && SortGroupIcon && <SortGroupIcon />}
            </LinkButton>
        );
    };

    render(): ReactNode {
        const {
            className,
            needSyncSortWithServer,
            sortInfo: {availableSortTypeGroups},
            size,
        } = this.props;

        return (
            <div
                className={cx(
                    'searchHotelsSortBar',
                    `searchHotelsSortBar_size_${size}`,
                    className,
                    {
                        searchHotelsSortBar_canPreventSelect:
                            needSyncSortWithServer,
                    },
                )}
                {...prepareQaAttributes(this.props)}
            >
                {availableSortTypeGroups.map(this.renderSortGroup)}
            </div>
        );
    }
}

export default SearchHotelsSortBar;
