import React, {PureComponent, ReactNode} from 'react';
import _flatMap from 'lodash/flatMap';

import {IWithClassName} from 'types/withClassName';
import {
    ISortInfo,
    ISortType,
    ISortTypeGroup,
    TSortId,
} from 'types/hotels/search/ISortInfo';
import {IWithDeviceType} from 'types/withDeviceType';
import {TSetActiveSortFunc} from '../../types/TSetActiveSortFunc';

import getSortGroupIcon from '../../utilities/getSortGroupIcon';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

/* Components */

import Select from 'components/Select/Select';

import cx from './HotelsSortSelect.scss';

/* Component Props */
export interface ISearchHotelsSortSelectProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    sortInfo: ISortInfo;
    onSetActiveSort: TSetActiveSortFunc;
    showIcon?: boolean;
}
interface ISortSelectOption {
    value: TSortId;
    data: ISortType;
}

/* Constants */
const MENU_MAX_HEIGHT = 300;

class SearchHotelsSortSelect extends PureComponent<ISearchHotelsSortSelectProps> {
    /* Handlers */
    private handleChangeSelectedSort = (sortId: TSortId): void => {
        const {
            sortInfo: {selectedSortId, availableSortTypeGroups},
            onSetActiveSort,
        } = this.props;

        const selectedSorеGroup = availableSortTypeGroups.find(sortGroup =>
            sortGroup.sortTypes.some(sortType => sortId === sortType.id),
        );

        if (selectedSortId !== sortId) {
            onSetActiveSort({
                id: sortId,
                requiresGeoLocation: selectedSorеGroup?.requiresGeoLocation,
            });
        }
    };

    /* Helpers */

    prepareSelectOptions = (): ISortSelectOption[] => {
        const {
            sortInfo: {availableSortTypeGroups},
        } = this.props;

        return _flatMap<ISortTypeGroup, ISortSelectOption>(
            availableSortTypeGroups,
            ({sortTypes}): ISortSelectOption[] =>
                sortTypes.map((sortType): ISortSelectOption => {
                    const {id} = sortType;

                    return {
                        value: id,
                        data: sortType,
                    };
                }),
        );
    };

    /* Render */
    renderSelectItem = (option: ISortSelectOption): string => {
        return option.data.name;
    };

    renderSelectValue = (option: ISortSelectOption): React.ReactElement => {
        const {
            sortInfo: {availableSortTypeGroups},
            showIcon,
            deviceType,
        } = this.props;
        const selectedSortTypeGroup = availableSortTypeGroups.find(
            group => group.id === option.data.id,
        );

        const Icon = getSortGroupIcon(
            deviceType,
            selectedSortTypeGroup?.sortTypes,
            option.data.id,
        );

        return (
            <div className={cx('sortSelectOption')}>
                {showIcon && Icon && (
                    <Icon
                        width="16"
                        height="16"
                        className={cx('sortSelectOptionIcon')}
                    />
                )}
                {option.data.name}
            </div>
        );
    };

    render(): ReactNode {
        const {
            sortInfo: {selectedSortId},
            className,
        } = this.props;

        return (
            <div className={cx(className)} {...prepareQaAttributes(this.props)}>
                <Select<TSortId, ISortType>
                    size="l"
                    width="max"
                    menuWidth="fixed"
                    onChange={this.handleChangeSelectedSort}
                    value={selectedSortId}
                    options={this.prepareSelectOptions()}
                    menuMaxHeight={MENU_MAX_HEIGHT}
                    renderItem={this.renderSelectItem}
                    renderValue={this.renderSelectValue}
                />
            </div>
        );
    }
}

export default SearchHotelsSortSelect;
