import {FunctionComponent, useCallback, useMemo, ChangeEvent} from 'react';
import _flatMap from 'lodash/flatMap';

import {
    ISortInfo,
    ISortType,
    ISortTypeGroup,
} from 'types/hotels/search/ISortInfo';
import {TSetActiveSortFunc} from '../../types/TSetActiveSortFunc';

import {useBoolean} from 'utilities/hooks/useBoolean';

import * as i18nCommonSort from 'i18n/common-sort';

import BottomSheet from 'components/BottomSheet/BottomSheet';
import Button from 'components/Button/Button';
import Radiobox, {RadioOptionProps} from 'components/Radiobox/Radiobox';
import Text from 'components/Text/Text';
import Skeleton from 'components/Skeletons/Skeleton/Skeleton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import ArrowBottomIcon from 'icons/16/ArrowBottom';

import cx from './SortBottomSheet.scss';

interface ISortBottomSheetProps {
    sortInfo: ISortInfo | undefined;
    onSetActiveSort: TSetActiveSortFunc;
}

const SortBottomSheet: FunctionComponent<ISortBottomSheetProps> = ({
    sortInfo,
    onSetActiveSort,
}) => {
    const {
        setTrue: setVisible,
        setFalse: setHidden,
        value: isVisible,
    } = useBoolean(false);

    const sortTypes = useMemo(() => {
        return _flatMap<ISortTypeGroup, ISortType>(
            sortInfo?.availableSortTypeGroups || [],
            (g): ISortType[] => g.sortTypes,
        );
    }, [sortInfo]);

    const options = useMemo(() => {
        return sortTypes.map(
            ({id, name}): RadioOptionProps => ({
                label: name,
                value: id,
                className: cx('item'),
            }),
        );
    }, [sortTypes]);

    const handleRadioChange = useCallback(
        (e: ChangeEvent<HTMLInputElement>) => {
            setHidden();

            const sortId = e.target.value;
            const selectedStoreGroup = sortInfo?.availableSortTypeGroups.find(
                sortGroup =>
                    sortGroup.sortTypes.some(
                        sortType => sortId === sortType.id,
                    ),
            );

            onSetActiveSort({
                id: sortId,
                requiresGeoLocation: selectedStoreGroup?.requiresGeoLocation,
            });
        },
        [setHidden, onSetActiveSort, sortInfo],
    );

    const selectedOption = options.find(
        g => g.value === sortInfo?.selectedSortId,
    );

    if (!sortInfo) {
        return <Skeleton className={cx('triggerSize')} withAnimation />;
    }

    return (
        <>
            <Button onClick={setVisible} className={cx('triggerSize')}>
                <TextWithIcon
                    text={selectedOption?.label}
                    iconRight={ArrowBottomIcon}
                />
            </Button>
            <BottomSheet isOpened={isVisible} onClose={setHidden}>
                <Text className={cx('title')} weight="bold" size="xl" tag="div">
                    {i18nCommonSort.title()}
                </Text>
                <Radiobox
                    className={cx('radio')}
                    value={sortInfo.selectedSortId}
                    options={options}
                    onChange={handleRadioChange}
                />
            </BottomSheet>
        </>
    );
};

export default SortBottomSheet;
