import React, {useMemo, useEffect} from 'react';
import _times from 'lodash/times';
import {useInView} from 'react-intersection-observer';

import cx from './LazyLoad.scss';

const DEFAULT_ITEMS_TO_LOAD = 10;

interface ILazyLoadProps {
    totalItemsCount?: number;
    placeholderNode?: React.ReactNode;
    placeholdersCount?: number;
    isLoading: boolean;
    onBottomScroll: () => void;
}

const LazyLoad: React.FC<ILazyLoadProps> = props => {
    const {
        children,
        totalItemsCount = Infinity,
        placeholderNode,
        placeholdersCount,
        isLoading,
        onBottomScroll,
    } = props;

    const hasItems = React.Children.count(children);
    const needPlaceholder = hasItems && isLoading && placeholderNode;

    const placeholderImages = useMemo(() => {
        const items: React.ReactNode[] = [];
        const placeholderCount =
            placeholdersCount ||
            Math.min(DEFAULT_ITEMS_TO_LOAD, totalItemsCount - items.length);

        _times(placeholderCount, i =>
            items.push(
                <React.Fragment key={`placeholder-${i}`}>
                    {placeholderNode}
                </React.Fragment>,
            ),
        );

        return items;
    }, [totalItemsCount, placeholderNode, placeholdersCount]);

    const [ref, inView] = useInView();

    useEffect(() => {
        if (inView) {
            onBottomScroll();
        }
    }, [inView, onBottomScroll]);

    return (
        <>
            {hasItems && children}
            {needPlaceholder && placeholderImages}
            <div
                className={cx('viewObserver', {viewObserver_inView: inView})}
                ref={ref}
            />
        </>
    );
};

export default LazyLoad;
