import React, {PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IBookOfferCancellationInfo} from 'reducers/depreacted/hotels/bookAndPayPage/offerInfoByToken/types';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {HUMAN} from 'utilities/dateUtils/formats';
import {prepareRefundableDate} from './utilities/getRefundableRule/getRefundableRule';
import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-OfferRefundableRules';

import MessageBoxPopup from 'components/MessageBoxPopup/MessageBoxPopup';
import BottomSheetWithTrigger from 'projects/depreacted/hotels/components/BottomSheetWithTrigger/BottomSheetWithTrigger';
import InfoIcon from 'icons/16/Info';
import CheckIcon from 'icons/16/CheckCircle';
import WarningFilledIcon from 'icons/16/WarningFilled';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

import OfferRefundableRules, {
    OFFER_REFUNDABLE_THEMES,
} from './OfferRefundableRules';

import cx from './OfferRefundableRulesWithTrigger.scss';

const CANCELLATION_INFO_QA = 'cancellationInfo';

const DIRECTION = [EPopupDirection.BOTTOM_RIGHT];

interface IOfferRefundableRulesWithTriggerProps
    extends IWithClassName,
        IWithQaAttributes {
    trigger: {
        showIcon: boolean;
        size: string;
    };
    labelTheme?: 'black';
    preventRenderDetailedInfo?: boolean;
    canRenderOnlyFullyRefundable?: boolean;
    cancellationInfo: IBookOfferCancellationInfo;
}

class OfferRefundableRulesWithTrigger extends PureComponent<IOfferRefundableRulesWithTriggerProps> {
    triggerRef = React.createRef<HTMLDivElement>();

    static defaultProps = {
        className: '',
        trigger: {
            showIcon: false,
            size: 's',
        },
        cancellationInfo: {
            refundable: false,
            refundableRules: [],
        },
    };

    state = {
        isPopupVisible: false,
    };

    hidePopup = (): void => {
        this.setState({isPopupVisible: false});
    };

    checkFullRefundableOffer = (): boolean => {
        const {
            cancellationInfo: {refundableRules},
        } = this.props;
        const refundablePeriod = refundableRules && refundableRules[0];

        return refundablePeriod?.type === 'NO_PENALTY';
    };

    togglePopup = (): void => {
        this.setState({isPopupVisible: !this.state.isPopupVisible});
    };

    renderTooltip = ({isMobile}: {isMobile: boolean}): ReactNode => {
        if (isMobile) {
            return (
                <BottomSheetWithTrigger
                    renderTriggerNode={this.renderBottomSheetTrigger}
                >
                    {this.renderRefundableRules(OFFER_REFUNDABLE_THEMES.white)}
                </BottomSheetWithTrigger>
            );
        }

        return (
            <>
                {this.renderTriggerNode({
                    ref: this.triggerRef,
                    onClick: this.togglePopup,
                    canRenderIcon: true,
                })}
                <MessageBoxPopup
                    anchorRef={this.triggerRef}
                    isVisible={this.state.isPopupVisible}
                    onClose={this.hidePopup}
                    direction={DIRECTION}
                    secondaryOffset={-20}
                    tailOffset={20}
                >
                    {this.renderContentNode()}
                </MessageBoxPopup>
            </>
        );
    };

    renderBottomSheetTrigger = ({
        input,
    }: {
        input: {onClick: () => void};
    }): ReactNode => {
        const {onClick} = input;

        return this.renderTriggerNode({
            onClick,
            canTitleClick: true,
            canRenderIcon: true,
        });
    };

    renderTriggerNode = (
        triggerProps: {
            ref?: React.RefObject<HTMLDivElement>;
            canTitleClick?: boolean;
            onClick?: React.MouseEventHandler;
            canRenderIcon?: boolean;
        } = {},
    ): ReactNode => {
        const {
            labelTheme,
            cancellationInfo: {refundableRules},
            trigger: {showIcon, size},
        } = this.props;
        const refundablePeriod = refundableRules[0];
        const isRefundableOffer = refundablePeriod?.type === 'NO_PENALTY';
        const refundableOfferDate = refundablePeriod?.endDate;

        const triggerTitle = isRefundableOffer
            ? i18nBlock.freeRuleTitleTrigger({
                  date: prepareRefundableDate(refundableOfferDate, HUMAN),
              })
            : i18nBlock.fullPriceTitleTrigger();

        const {canTitleClick, onClick} = triggerProps;

        return (
            <div
                className={cx('trigger', `trigger_size_${size}`, {
                    trigger_refundable: isRefundableOffer,
                    [`trigger_theme_${labelTheme}`]: labelTheme,
                })}
            >
                {showIcon && (
                    <span className={cx('triggerIcon')}>
                        {isRefundableOffer ? (
                            <CheckIcon />
                        ) : (
                            <WarningFilledIcon />
                        )}
                    </span>
                )}
                <span
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: `${CANCELLATION_INFO_QA}-trigger`,
                    })}
                    onClick={canTitleClick ? onClick : undefined}
                >
                    {triggerTitle}
                </span>
                {this.renderAttentionIcon(triggerProps)}
            </div>
        );
    };

    renderAttentionIcon = ({
        ref,
        onClick,
        canRenderIcon,
    }: {
        ref?: React.RefObject<HTMLDivElement>;
        canRenderIcon?: boolean;
        onClick?: React.MouseEventHandler;
    } = {}): ReactNode => {
        return (
            canRenderIcon && (
                <div
                    ref={ref}
                    className={cx('attentionIcon')}
                    onClick={onClick}
                >
                    <InfoIcon />
                </div>
            )
        );
    };

    renderRefundableRules = (
        theme = OFFER_REFUNDABLE_THEMES.dark,
    ): React.ReactNode => {
        const {cancellationInfo} = this.props;

        return (
            <OfferRefundableRules
                theme={theme}
                cancellationInfo={cancellationInfo}
            />
        );
    };

    renderContentNode = (): ReactNode => {
        return (
            <div className={cx('content')}>{this.renderRefundableRules()}</div>
        );
    };

    render(): ReactNode {
        const {
            className,
            preventRenderDetailedInfo,
            canRenderOnlyFullyRefundable,
            cancellationInfo: {refundableRules},
        } = this.props;
        const isEmptyRulesList = refundableRules.length === 0;
        const isFullRefundableOffer = this.checkFullRefundableOffer();

        if (canRenderOnlyFullyRefundable && !isFullRefundableOffer) {
            return null;
        }

        return (
            <DeviceTypeContext.Consumer>
                {(deviceType): ReactNode => (
                    <div className={cx('offerRefundableRules', className)}>
                        {isEmptyRulesList || preventRenderDetailedInfo
                            ? this.renderTriggerNode()
                            : this.renderTooltip(deviceType)}
                    </div>
                )}
            </DeviceTypeContext.Consumer>
        );
    }
}

export default OfferRefundableRulesWithTrigger;
