import React, {
    ChangeEvent,
    useCallback,
    useEffect,
    useMemo,
    useRef,
    useState,
} from 'react';
import {useDispatch, useSelector} from 'react-redux';

import {HOTELS_SEARCH_FORM_NAME} from 'constants/reduxForm';

import {IWithClassName} from 'types/withClassName';
import {
    ESearchFormFieldName,
    ESearchFormSize,
    ESearchFormTriggerViewType,
} from 'components/SearchForm/types';
import {HotelSlugType} from 'types/hotels/hotel/IHotel';
import {ISearchDebugQueryParams} from 'types/hotels/common/IQueryParams';
import {
    ESearchSuggestItemType,
    IHotelsSuggest,
} from 'types/hotels/common/ISearchSuggest';
import {ISearchFormRenderTravelersBlockProps} from 'components/SearchForm/types/ISearchFormRenderTravelersBlockProps';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {ECalendarType} from 'components/Calendar/types';
import {ISelectItemPreparedOptions} from 'components/SearchForm/types/ISelectItemOptions';
import {IHotelsPreviousSearch} from 'types/hotels/previousSearches/IPreviousSearch';

import {
    setHotelsSearchFormAdultsFieldAction,
    setHotelsSearchFormChildrenAgesFieldAction,
    setHotelsSearchFormEndDateFieldAction,
    setHotelsSearchFormStartDateFieldAction,
    setHotelsSearchFormToFieldAction,
} from 'reducers/depreacted/hotels/searchForm/actions';
import {
    fetchAndSelectSearchSuggestAction,
    fetchSearchSuggestActions,
    logSearchSuggestAction,
} from 'reducers/depreacted/hotels/searchSuggests/actions';
import {openHotelsGeolocationErrorModalAction} from 'reducers/common/hotelsGeolocationErrorModal/actions';
import fillSearchFormByPreviousSearchThunkAction from 'reducers/depreacted/hotels/previousSearches/thunk/fillSearchFormByPreviousSearch';
import restorePreviousSearchesThunkAction from 'reducers/depreacted/hotels/previousSearches/thunk/restorePreviousSearchesThunkAction';
import {clearLastSearchTimestampAction} from 'reducers/depreacted/hotels/searchPage/search/actions';

import searchFormSelector from 'selectors/depreacted/hotels/searchForm/searchFormSelector';

import useDispatchedAction from 'utilities/hooks/useDispatchedAction';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import useImmutableCallback from 'utilities/hooks/useImmutableCallback';
import getSuggestTitleAndDescription from 'projects/depreacted/hotels/components/SearchForm/utilities/getSuggestTitleAndDescription';
import validateForm from 'projects/depreacted/hotels/components/SearchForm/utilities/validateForm';
import {getQueryByBrowserHistory} from 'utilities/getQueryByBrowserHistory/getQueryByBrowserHistory';
import {reachGoal} from 'utilities/metrika';
import getHotelsGeolocationSuggestMetrika from 'projects/depreacted/hotels/utilities/metrika/getHotelsGeolocationSuggestMetrika';
import browserGeolocation from 'utilities/browserGeolocation/browserGeolocation';
import {useBoolean} from 'utilities/hooks/useBoolean';
import renderSuggestItemDescription from 'projects/depreacted/hotels/components/SearchForm/utilities/renderSuggestItemDescription';
import renderSuggestItemTitle from 'projects/depreacted/hotels/components/SearchForm/utilities/renderSuggestItemTitle';
import {
    getQa,
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/components';
import * as i18nSearchForm from 'i18n/hotels-SearchForm';

import CommonSearchForm, {
    IBaseSearchFormProps,
    IOnDateClickParams,
} from 'components/SearchForm/SearchForm';
import useRequestSuggestsWithThrottle from 'components/SearchForm/hooks/useRequestSuggestsWithThrottle';
import useHandleSubmit, {
    ISubmitArgs,
} from 'projects/depreacted/hotels/components/SearchForm/hooks/useHandleSubmit';
import useUpdateBySearchInformation from 'projects/depreacted/hotels/components/SearchForm/hooks/useUpdateBySearchInformation';
import ProductionOffersCheckbox from 'projects/depreacted/hotels/components/SearchForm/components/ProductionOffersCheckbox/ProductionOffersCheckbox';
import HotelsTravellers from 'projects/depreacted/hotels/components/SearchForm/components/HotelsTravellers/HotelsTravellers';
import {useNewCalendarEnabled} from 'components/DatePicker/hooks/useNewCalendarEnabled';
import {
    ESuggestSource,
    ISuggestValue,
} from 'components/SearchSuggest/SearchSuggest';
import {IDatePickerFooterProps} from 'components/DatePicker/components/DatePickerFooter/DatePickerFooter';
import useCalendarConversionExpMetrics from 'components/DatePicker/hooks/useCalendarConversionExpMetrics';

const FIELD_NAMES = [
    ESearchFormFieldName.TO,
    ESearchFormFieldName.START_DATE,
    ESearchFormFieldName.END_DATE,
];

export interface ISearchFormProps
    extends IBaseSearchFormProps,
        IWithClassName,
        IWithQaAttributes {
    activeHotelSlug?: HotelSlugType;
    hasDirectionSuggest?: boolean;
    hasDelayBeforeUpdateLocation?: boolean;
    queryParams?: Record<string, string>;
    searchFormClassName?: string;
    submitButtonClassName?: string;
    travellersClassName?: string;
    travellersTriggerClassName?: string;
    datePickerTriggerClassName?: string;
    formRef?: React.RefObject<HTMLFormElement>;
    onSubmit?(params: {searchLink: string}): void;
    onChangeToPoint?(value: ISuggestValue<IHotelsSuggest>): void;
}

const SearchForm: React.FC<ISearchFormProps> = props => {
    const {
        className,
        searchFormClassName,
        submitButtonClassName,
        travellersClassName,
        travellersTriggerClassName,
        datePickerTriggerClassName,
        onInteract,
        isExpanded,
        size = ESearchFormSize.XL,
        triggerViewType = ESearchFormTriggerViewType.UNION,
        activeHotelSlug,
        hasDirectionSuggest = true,
        hasDelayBeforeUpdateLocation = false,
        canSubmitOnChange = false,
        canToggleDropdowns = false,
        fieldsRef,
        onShowFieldPopup,
        onHideFieldPopup,
        onSubmit,
        queryParams,
        formRef,
        onChangeToPoint,
        ...restProps
    } = props;

    const rootQa = getQa(props);

    const {
        toField,
        startDateField,
        endDateField,
        adultsField,
        childrenAgesField,
        searchSuggests,
        searchInformation,
        enableDebugHotelProdOffers,
        userGeoLocation,
    } = useSelector(searchFormSelector);

    const isGeolocationSuggestExists = searchSuggests.items.some(
        item =>
            item?.redirectParams?.type === ESearchSuggestItemType.HOTELS_NEARBY,
    );

    const dispatch = useDispatch();
    const deviceType = useDeviceType();
    const {
        value: isSuggestLoading,
        setTrue: startSuggestLoading,
        setFalse: stopSuggestLoading,
    } = useBoolean(false);

    const [hasProductionOffers, setHasProductionOffers] = useState(false);

    const chosenPreviousSearch = useRef<IHotelsPreviousSearch | null>(null);
    const chosenPersonalSuggest = useRef<IHotelsPreviousSearch | null>(null);

    const setToPoint = useImmutableCallback(
        (fieldValue: ISuggestValue<IHotelsSuggest>) => {
            dispatch(setHotelsSearchFormToFieldAction(fieldValue));

            onChangeToPoint?.(fieldValue);
        },
    );
    const setStartDate = useDispatchedAction(
        setHotelsSearchFormStartDateFieldAction,
        dispatch,
    );
    const setEndDate = useDispatchedAction(
        setHotelsSearchFormEndDateFieldAction,
        dispatch,
    );
    const setAdults = useDispatchedAction(
        setHotelsSearchFormAdultsFieldAction,
        dispatch,
    );
    const setChildrenAges = useDispatchedAction(
        setHotelsSearchFormChildrenAgesFieldAction,
        dispatch,
    );

    const fetchAndSelectSearchSuggest = useDispatchedAction(
        fetchAndSelectSearchSuggestAction,
        dispatch,
    );

    const openHotelsGeolocationErrorModal = useDispatchedAction(
        openHotelsGeolocationErrorModalAction,
        dispatch,
    );

    const clearLastSearchTimestamp = useDispatchedAction(
        clearLastSearchTimestampAction,
        dispatch,
    );

    const {reachSearchInteraction, reachCalendarInteraction} =
        useCalendarConversionExpMetrics(EHotelsGoal);

    const requestSuggests = useImmutableCallback((fieldInputValue: string) => {
        dispatch(
            fetchSearchSuggestActions.request({
                fieldType: ESearchFormFieldName.TO,
                fetchParams: {
                    inputValue: fieldInputValue,
                },
            }),
        );
    });

    const requestSuggestsWithThrottle =
        useRequestSuggestsWithThrottle(requestSuggests);

    const handleSuggestInputFocus = useCallback(() => {
        if (deviceType.isMobile) {
            requestSuggestsWithThrottle('');
        }
    }, [requestSuggestsWithThrottle, deviceType]);

    const handleTravelersFocus = useImmutableCallback(() => {
        // Должно срабатывать после скрытия других попапов
        requestAnimationFrame(() => {
            onInteract?.(true);
        });
    });

    const getTravelersBlurHandler = useImmutableCallback(submit => {
        return (): void => {
            onInteract?.(false);

            if (canSubmitOnChange) {
                requestAnimationFrame(() => {
                    submit();
                });
            }
        };
    });

    const handleSubmit = useHandleSubmit({
        toField,
        startDateField,
        endDateField,
        adultsField,
        childrenAgesField,
        activeHotelSlug,
        queryParams,
        hasProductionOffers,
        hasDirectionSuggest,
        hasDelayBeforeUpdateLocation,
        dispatch,
        chosenPreviousSearch,
        chosenPersonalSuggest,
        onSubmit,
    });

    const handleFormSubmit = useImmutableCallback((args: ISubmitArgs) => {
        clearLastSearchTimestamp();
        handleSubmit(args);
        reachSearchInteraction();

        if (
            hasDirectionSuggest &&
            toField.selectedValue &&
            userGeoLocation.geoId === toField.selectedValue.redirectParams.geoId
        ) {
            reachGoal(EHotelsGoal.HOTELS_SEARCH_MY_REGION);
        }
    });

    const handleChangeProductionOffersCheckbox = useImmutableCallback(
        (e: ChangeEvent<HTMLInputElement>) => {
            setHasProductionOffers(e.target.checked);
        },
    );

    const handleSelectSuggestItem = useImmutableCallback(
        async (
            item: IHotelsSuggest,
            {
                isUserInput,
                isManualClick,
                isTrustedUser,
            }: ISelectItemPreparedOptions,
        ) => {
            if (
                item.redirectParams.type ===
                    ESearchSuggestItemType.CROSS_SALE ||
                item.redirectParams.type ===
                    ESearchSuggestItemType.CROSS_SEARCH ||
                item.redirectParams.type === ESearchSuggestItemType.HISTORY
            ) {
                if (item.redirectParams.offerSearchParams) {
                    const suggest = {
                        to: item,
                        startDate:
                            item.redirectParams.offerSearchParams?.checkinDate,
                        endDate:
                            item.redirectParams.offerSearchParams?.checkoutDate,
                        adults: item.redirectParams.offerSearchParams?.adults,
                        childrenAges:
                            item.redirectParams.offerSearchParams?.childrenAges,
                    };

                    dispatch(
                        fillSearchFormByPreviousSearchThunkAction(suggest),
                    );

                    if (
                        item.redirectParams.type ===
                            ESearchSuggestItemType.CROSS_SALE ||
                        item.redirectParams.type ===
                            ESearchSuggestItemType.CROSS_SEARCH
                    ) {
                        chosenPersonalSuggest.current = suggest;
                        reachGoal(EHotelsGoal.HOTELS_SUGGEST_CROSSALE_CLICK);
                    } else {
                        reachGoal(EHotelsGoal.PREVIOUS_SEARCHES_SUGGEST_CLICK);
                    }
                }

                if (!item.redirectParams.offerSearchParams?.checkinDate) {
                    setTimeout(() => {
                        fieldsRef?.current?.focusFieldByName(
                            ESearchFormFieldName.START_DATE,
                        );
                    });
                }
            }

            requestAnimationFrame(() => {
                dispatch(
                    logSearchSuggestAction({
                        selectedId: item.id,
                        isUserInput,
                        isManualClick,
                        isTrustedUser,
                    }),
                );
            });

            if (
                item.redirectParams?.type ===
                ESearchSuggestItemType.HOTELS_NEARBY
            ) {
                startSuggestLoading();

                reachGoal(getHotelsGeolocationSuggestMetrika());

                const data = await browserGeolocation.getGeolocation();

                stopSuggestLoading();

                if (!data?.sortOrigin) {
                    setToPoint({
                        selectedValue: false,
                        inputValue: '',
                        source: ESuggestSource.INPUT,
                    });

                    return openHotelsGeolocationErrorModal?.();
                }

                fetchAndSelectSearchSuggest?.({
                    userCoordinates: data.sortOrigin,
                    geoLocationStatus: data.geoLocationStatus,
                    canSelectNearbyHotel: true,
                });
            }
        },
    );
    const isNewCalendarEnabled = useNewCalendarEnabled();

    const handleShowCalendar = useImmutableCallback(
        (calendarType: ECalendarType) => {
            const oldStartCalendarOpened =
                !isNewCalendarEnabled &&
                calendarType === ECalendarType.StartDate;
            const startCalendarOpened =
                oldStartCalendarOpened || isNewCalendarEnabled;

            if (!startCalendarOpened) {
                return;
            }

            reachGoal(EHotelsGoal.CALENDAR_OPEN);

            if (toField.selectedValue) {
                reachCalendarInteraction();
            }
        },
    );

    const handleDateClick = useImmutableCallback(
        (args?: IOnDateClickParams) => {
            if (!args) {
                return;
            }

            const {startDate, endDate, calendarType} = args;

            if (startDate && endDate && Number(startDate) === Number(endDate)) {
                if (
                    calendarType === ECalendarType.StartDate ||
                    isNewCalendarEnabled
                ) {
                    setEndDate(null);
                } else {
                    setStartDate(null);
                }
            }

            if (startDate && !endDate) {
                reachGoal(EHotelsGoal.CALENDAR_SELECT_CHECKIN_DATE);
            }

            if (startDate && endDate) {
                reachGoal(EHotelsGoal.CALENDAR_SELECT_CHECKOUT_DATE);
            }
        },
    );

    useEffect(() => {
        if (!hasDirectionSuggest) {
            return;
        }

        requestSuggestsWithThrottle(toField.inputValue);
    }, [hasDirectionSuggest, requestSuggestsWithThrottle, toField.inputValue]);

    useUpdateBySearchInformation(searchInformation, dispatch);

    useEffect(() => {
        const {debugUseProdOffers} =
            getQueryByBrowserHistory() as ISearchDebugQueryParams;

        setHasProductionOffers(Boolean(debugUseProdOffers));
    }, []);

    useEffect(() => {
        dispatch(restorePreviousSearchesThunkAction());
    }, [dispatch]);

    const renderTravellersBlock = useCallback(
        ({baseCx, submit}: ISearchFormRenderTravelersBlockProps) => {
            return (
                <div className={baseCx('travellers', travellersClassName)}>
                    <HotelsTravellers
                        triggerClassName={baseCx(
                            'travellersTrigger',
                            'lastControl',
                            travellersTriggerClassName,
                        )}
                        triggerFocusClassName={baseCx(
                            'travellersTrigger_focus',
                        )}
                        size={size}
                        triggerViewType={triggerViewType}
                        tabIndex={isExpanded ? 0 : -1}
                        adultsCount={adultsField}
                        setAdults={setAdults}
                        childrenAges={childrenAgesField}
                        setChildrenAges={setChildrenAges}
                        deviceType={deviceType}
                        canToggle={canToggleDropdowns}
                        onShowPopup={onShowFieldPopup}
                        onHidePopup={onHideFieldPopup}
                        onFocus={handleTravelersFocus}
                        onBlur={getTravelersBlurHandler(submit)}
                        {...prepareQaAttributes(rootQa)}
                    />
                </div>
            );
        },
        [
            adultsField,
            canToggleDropdowns,
            childrenAgesField,
            deviceType,
            getTravelersBlurHandler,
            handleTravelersFocus,
            isExpanded,
            onHideFieldPopup,
            onShowFieldPopup,
            rootQa,
            setAdults,
            setChildrenAges,
            size,
            travellersClassName,
            travellersTriggerClassName,
            triggerViewType,
        ],
    );

    const footerBlockParams = useMemo((): IDatePickerFooterProps => {
        let messageText;

        if (!startDateField) {
            messageText = i18nSearchForm.selectCheckInDate();
        } else if (!endDateField) {
            messageText = i18nSearchForm.selectCheckOutDate();
        }

        const buttonText = i18nSearchForm.selectButton();
        const visible = deviceType.isMobile;
        const showButton = true;
        const buttonDisabled = !startDateField || !endDateField;

        return {visible, messageText, showButton, buttonText, buttonDisabled};
    }, [startDateField, endDateField, deviceType.isMobile]);

    return (
        <div className={className}>
            <CommonSearchForm
                className={searchFormClassName}
                submitButtonClassName={submitButtonClassName}
                datePickerTriggerClassName={datePickerTriggerClassName}
                size={size}
                triggerViewType={triggerViewType}
                searchFormName={HOTELS_SEARCH_FORM_NAME}
                fieldsNames={FIELD_NAMES}
                toField={hasDirectionSuggest ? toField : undefined}
                startDateField={startDateField}
                endDateField={endDateField}
                hasEndDate
                toSearchSuggests={searchSuggests.items}
                defaultAutoSelectIndex={isGeolocationSuggestExists ? 1 : 0}
                validateForm={validateForm}
                uniqueSuggestValueName="id"
                getSuggestTitleAndDescription={getSuggestTitleAndDescription}
                renderSuggestItemTitle={renderSuggestItemTitle}
                renderSuggestItemDescription={renderSuggestItemDescription}
                toPointPlaceholder={i18nBlock.hotelsSearchFormDotToFieldPlaceholder()}
                startDatePlaceholder={i18nBlock.hotelsSearchFormDotStartDatePlaceholder()}
                endDatePlaceholder={i18nBlock.hotelsSearchFormDotEndDatePlaceholder()}
                setToPoint={hasDirectionSuggest ? setToPoint : undefined}
                setStartDate={setStartDate}
                setEndDate={setEndDate}
                onSubmit={handleFormSubmit}
                isRoundTrip
                isExpanded={isExpanded}
                renderTravellersBlock={renderTravellersBlock}
                canSubmitOnChange={canSubmitOnChange}
                fieldsRef={fieldsRef}
                onSelectSuggestItem={handleSelectSuggestItem}
                canToggleDropdowns={canToggleDropdowns}
                onShowFieldPopup={onShowFieldPopup}
                onHideFieldPopup={onHideFieldPopup}
                onShowCalendar={handleShowCalendar}
                onDateClick={handleDateClick}
                isSuggestLoading={isSuggestLoading}
                onInteract={onInteract}
                onSuggestInputFocus={handleSuggestInputFocus}
                formRef={formRef}
                datePickerFooterBlockParams={footerBlockParams}
                canAutoHideCalendar={
                    isNewCalendarEnabled ? !deviceType.isMobile : undefined
                }
                {...restProps}
            />

            {enableDebugHotelProdOffers && (
                <ProductionOffersCheckbox
                    checked={hasProductionOffers}
                    onChange={handleChangeProductionOffersCheckbox}
                />
            )}
        </div>
    );
};

export default React.memo(SearchForm);
