import React, {useEffect, useCallback, useRef} from 'react';
import {useSelector, useDispatch} from 'react-redux';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {StoreInterface} from 'reducers/storeTypes';
import {setSnackbarInfo} from 'reducers/common/snackbar/actions';
import {
    addFavoriteHotel,
    IAddFavoriteHotelParams,
} from 'reducers/depreacted/hotels/searchPage/search/thunkActions';
import {addFavoriteHotel as addFavoriteHotelOnHotelPage} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/thunkActions';

import {deviceMods} from 'utilities/stylesUtils';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import Flex from 'components/Flex/Flex';
import LinkButton from 'components/LinkButton/LinkButton';
import Text from 'components/Text/Text';
import BaseSnackbar from 'components/Snackbar/Snackbar';

import cx from './Snackbar.scss';

export interface ISnackbarProps extends IWithClassName, IWithDeviceType {
    currentPage: 'hotel' | 'favorites' | 'search' | null;
}

const Snackbar: React.FC<ISnackbarProps> = ({
    className,
    deviceType,
    currentPage,
}) => {
    const {
        lastAction,
        message,
        hasCancellation,
        cancelAction,
        cancelActionData,
        page,
    } = useSelector((state: StoreInterface) => state.common.snackbar);
    const ref = useRef<number>();
    const dispatch = useDispatch();
    const {isIos} = {isIos: false};

    useEffect(() => {
        if (lastAction) {
            clearTimeout(ref.current);

            ref.current = window.setTimeout(() => {
                dispatch(
                    setSnackbarInfo?.({
                        lastAction: null,
                        hasCancellation: false,
                        page: null,
                    }),
                );
            }, 5000);
        }

        return (): void => clearTimeout(ref.current);
    }, [lastAction, dispatch]);

    const onCancelClick = useCallback(() => {
        if (cancelAction && cancelActionData) {
            if (cancelAction === 'addFavoriteHotelOnSearch') {
                dispatch(
                    addFavoriteHotel(
                        cancelActionData as IAddFavoriteHotelParams,
                    ),
                );
            }

            if (cancelAction === 'addFavoriteHotelOnFavoritesPage') {
                // dispatch(
                //     addFavoriteHotelAction(
                //         cancelActionData as IAddFavoriteHotelRequestParams,
                //     ),
                // );
            }

            if (cancelAction === 'addFavoriteHotelOnHotelPage') {
                dispatch(
                    addFavoriteHotelOnHotelPage(
                        cancelActionData as IAddFavoriteHotelParams,
                    ),
                );
            }

            dispatch(
                setSnackbarInfo?.({
                    lastAction: null,
                    hasCancellation: false,
                    page: null,
                }),
            );
        }
    }, [cancelAction, cancelActionData, dispatch]);

    return (
        <BaseSnackbar
            isVisible={Boolean(lastAction && page && page === currentPage)}
            className={cx('root', deviceMods('root', deviceType), className, {
                root_IOS: isIos,
            })}
        >
            <Flex>
                {message && (
                    <Text color="inverse" className={cx('content')}>
                        {message}
                    </Text>
                )}
                {hasCancellation && (
                    <LinkButton
                        className={cx('cancelLink')}
                        onClick={onCancelClick}
                        theme="brand"
                    >
                        {i18nFavoritesBlock.cancel()}
                    </LinkButton>
                )}
            </Flex>
        </BaseSnackbar>
    );
};

export default Snackbar;
