import React, {FunctionComponent, useMemo} from 'react';

import {IWithClassName} from 'types/withClassName';
import {
    ITransportStation,
    TransportStationType,
} from 'types/hotels/hotel/ITransportStation';
import {IIconProps} from 'icons/types/icon';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModMobile} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import TextWithIcon, {TSize} from 'components/TextWithIcon/TextWithIcon';
import TransportStopIcon12 from 'icons/12/TransportStop';
import TransportStopIcon16 from 'icons/16/TransportStop';
import Text from 'components/Text/Text';

import cx from './TransportAccessibility.scss';

export interface ITransportAccessibilityProps
    extends IWithClassName,
        IWithQaAttributes {
    textClassName?: string;
    iconClassName?: string;
    station: ITransportStation;
    textWithIconSize?: TSize;
    bigIcon?: boolean;
}

/**
 * Остановка или станция метро
 */
const TransportAccessibility: FunctionComponent<ITransportAccessibilityProps> =
    ({
        className,
        textClassName,
        station,
        textWithIconSize,
        bigIcon,
        iconClassName,
        ...rest
    }) => {
        const deviceType = useDeviceType();

        const text = useMemo(
            () => (
                <>
                    <span className={cx('stationName')} title={station.name}>
                        {station.name}
                    </span>
                    <span
                        className={cx('distanceToStation')}
                        {...prepareQaAttributes({
                            parent: rest,
                            current: 'distance',
                        })}
                    >
                        {station.distanceText}
                    </span>
                </>
            ),
            [station.distanceText, station.name],
        );

        let Icon: React.ComponentType<IIconProps>;

        switch (station.type) {
            case TransportStationType.METRO: {
                const metroLineColor = station?.metroLine?.color;

                if (metroLineColor) {
                    return (
                        <div
                            className={cx(
                                'circleWithText',
                                `circleWithText_${bigIcon ? 'big' : 'normal'}`,
                                'station',
                                deviceModMobile('station', deviceType),
                                className,
                            )}
                            {...prepareQaAttributes(rest)}
                        >
                            <div
                                className={cx('circle')}
                                style={{backgroundColor: metroLineColor}}
                            />
                            <Text
                                className={cx('text', textClassName)}
                                size={textWithIconSize}
                            >
                                {text}
                            </Text>
                        </div>
                    );
                }

                Icon = bigIcon ? TransportStopIcon16 : TransportStopIcon12;

                break;
            }
            case TransportStationType.OTHER: {
                Icon = bigIcon ? TransportStopIcon16 : TransportStopIcon12;

                break;
            }
        }

        return (
            <TextWithIcon
                className={cx(
                    'station',
                    deviceModMobile('station', deviceType),
                    className,
                )}
                textClassName={cx('text', textClassName)}
                iconLeft={Icon}
                iconLeftClassName={cx('iconLeft', iconClassName)}
                size={textWithIconSize}
                text={text}
                spaceBetween={1}
                {...prepareQaAttributes(rest)}
            />
        );
    };

export default TransportAccessibility;
