import React, {ComponentType, ReactNode} from 'react';
import {connect} from 'react-redux';
import {withRouter, RouteComponentProps} from 'react-router-dom';
import _noop from 'lodash/noop';
import _isUndefined from 'lodash/isUndefined';
import _flow from 'lodash/flow';

import {INITIAL} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

import {IWithDeviceType} from 'types/withDeviceType';

import {fetchOfferInfoByTokenAction} from 'reducers/depreacted/hotels/bookAndPayPage/offerInfoByToken/actions';
import {
    getOrderInfoActions,
    stopOrderInfoPollingAction,
} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/actions';

import hotelsBookInfoSelector from 'selectors/depreacted/hotels/book/hotelsBookInfoSelector';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {
    getHotelOfferToken,
    getOfferAttributionParams,
} from 'projects/depreacted/hotels/utilities/getHotelPageParams/getOfferAttributionParamsByLocation';

import DeviceTypeContext from 'contexts/DeviceTypeContext';

interface IProviderSettings {
    canFetchOfferInfo: boolean;
    canStartOrderPolling: boolean;
}

const DEFAULT_PROVIDER_SETTINGS: IProviderSettings = {
    canFetchOfferInfo: false,
    canStartOrderPolling: false,
};

const mapStateToProps = hotelsBookInfoSelector;
const mapDispatchToProps = {
    fetchOfferInfoByToken: fetchOfferInfoByTokenAction.request,
    startOrderPolling: getOrderInfoActions.request,
    stopOrderPolling: stopOrderInfoPollingAction,
};

const bookInfoProviderContainer = _flow(
    withRouter,
    connect(mapStateToProps, mapDispatchToProps),
);

type TBookInfoProviderStateContainer = ReturnType<typeof mapStateToProps>;
type TBookInfoProviderDispatchContainer = typeof mapDispatchToProps;

type TBookInfoProviderContainerProps = RouteComponentProps &
    TBookInfoProviderStateContainer &
    TBookInfoProviderDispatchContainer;

interface IBookInfoProviderProps extends TBookInfoProviderContainerProps {
    orderId: string;
}

export interface IBookInfoComponentProps
    extends IBookInfoProviderProps,
        IWithDeviceType {}

export default (
        providerSettings: IProviderSettings = DEFAULT_PROVIDER_SETTINGS,
    ) =>
    (PageComponent: ComponentType<any>): ComponentType<any> => {
        return bookInfoProviderContainer(
            class BookInfoComponent extends React.PureComponent<IBookInfoProviderProps> {
                static defaultProps = {
                    offerInfoByToken: {},
                    orderInfo: {},
                    userInfo: {},
                    voucherInfo: {},
                    fetchOfferInfoByToken: _noop,
                    startOrderPolling: _noop,
                };

                componentDidMount(): void {
                    const {canFetchOfferInfo, canStartOrderPolling} =
                        providerSettings;

                    if (canFetchOfferInfo) {
                        this.fetchOfferInfoByToken();
                    }

                    if (canStartOrderPolling) {
                        this.startOrderPolling();
                    }
                }

                private fetchOfferInfoByToken(): void {
                    const {
                        fetchOfferInfoByToken,
                        location,
                        orderInfo: {status: orderStatus},
                        orderId: orderIdFromProps,
                    } = this.props;

                    const query = getQueryByLocation(location);

                    const {orderId: orderIdFromLocation} = query;
                    const {token} = getHotelOfferToken(query);
                    const {label} = getOfferAttributionParams(query);

                    const orderId = orderIdFromProps || orderIdFromLocation;

                    /* Fetch offer info for empty orderId by token and label from query params */
                    if (
                        orderStatus === INITIAL &&
                        token &&
                        _isUndefined(orderId)
                    ) {
                        fetchOfferInfoByToken({token, label});
                    }
                }

                private startOrderPolling(): void {
                    const {
                        startOrderPolling,
                        location,
                        orderId: orderIdFromProps,
                    } = this.props;

                    const {orderId: orderIdFromLocation} =
                        getQueryByLocation(location);

                    const orderId = orderIdFromProps || orderIdFromLocation;

                    /* Fetch order info by orderId from query params */
                    if (orderId && typeof orderId === 'string') {
                        startOrderPolling({orderId});
                    }
                }

                render(): React.ReactNode {
                    return (
                        <DeviceTypeContext.Consumer>
                            {(deviceType): ReactNode => {
                                const pageComponentProps = {
                                    deviceType,
                                    ...this.props,
                                };

                                return (
                                    <PageComponent {...pageComponentProps} />
                                );
                            }}
                        </DeviceTypeContext.Consumer>
                    );
                }
            },
        );
    };
