import {Component, ReactNode} from 'react';
import {History, Location} from 'history';

import {
    CANCELLED,
    CANCELLED_WITH_REFUND,
    CONFIRMED,
    FAILED,
    NEED_AUTHORIZATION,
    OFFER_FETCHED,
    OFFER_IS_NOT_AVAILABLE,
    OFFER_NOT_FOUND_ERROR,
    REFUND_FAILED,
    REFUNDED,
    UNKNOWN_OFFER_ERROR,
    UNKNOWN_ORDER_ERROR,
    WRONG_OFFER_TOKEN,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';
import {URLs} from 'constants/urls';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {TMetrikaHotelsParams} from 'utilities/metrika/types/params/hotels';

import {getInvalidInputDetailsFieldType} from 'selectors/depreacted/hotels/book/orderCancellationDetails/getInvalidInputDetailsFieldType';
import {TOrderInfoSelector} from 'selectors/depreacted/hotels/book/getBookOrderInfo';
import {TOfferInfoByTokenSelector} from 'selectors/depreacted/hotels/book/getBookOfferInfo';

import {reachGoal} from 'utilities/metrika';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {internalUrl} from 'utilities/url';
import {hotelsURLs} from 'projects/depreacted/hotels/utilities/urls';

import bookInfoProvider from 'projects/depreacted/hotels/containers/BookInfoProvider/BookInfoProvider';

interface ITokenPageQuery {
    token?: string | null;
    label?: string | null;
}

interface IOrderPageQuery {
    orderId?: string | null;
}

type TPageQuery = IOrderPageQuery | ITokenPageQuery | undefined;

interface IBookRedirectControllerProps {
    orderInfo: TOrderInfoSelector;
    offerInfoByToken: TOfferInfoByTokenSelector;
    history: History;
    location: Location;
}

class BookRedirectController extends Component<IBookRedirectControllerProps> {
    static defaultProps = {
        orderInfo: {},
        offerInfoByToken: {},
        location: {},
    };

    componentDidMount(): void {
        this.checkConfirmedOrderRedirect();
    }

    componentDidUpdate(prevProps: IBookRedirectControllerProps): void {
        this.checkRedirectToPageByStatus(prevProps);
    }

    checkConfirmedOrderRedirect(): void {
        const {
            orderInfo: {orderId, status: orderStatus, orderCancellationDetails},
            history,
            location,
        } = this.props;

        if (orderStatus === CONFIRMED) {
            const path = hotelsURLs.getPathByOrderStatus({
                orderId,
                orderStatus,
                orderCancellationDetails,
                location,
            });

            if (path) {
                history.replace(path);
            }
        }
    }

    checkRedirectToPageByStatus(prevProps: IBookRedirectControllerProps): void {
        const {history} = this.props;
        const hotelsBookPagePath =
            this.checkOrderStatus(prevProps) ||
            this.checkOfferStatus(prevProps);

        if (hotelsBookPagePath) {
            history.push(hotelsBookPagePath);
        }
    }

    getHotelsPath(
        nextPagePath: string,
        pageBaseQuery: TPageQuery,
    ): string | undefined {
        const {
            location: {pathname},
        } = this.props;

        if (this.isSamePathname(pathname, nextPagePath)) {
            return internalUrl(nextPagePath, pageBaseQuery);
        }

        return undefined;
    }

    isSamePathname(prevPathname: string, nextPagePath: string): boolean {
        return Boolean(
            nextPagePath && internalUrl(nextPagePath) !== prevPathname,
        );
    }

    getOrderIdByLocation(): IOrderPageQuery {
        const {location} = this.props;
        const {orderId} = getQueryByLocation(location) as IOrderPageQuery;

        return {
            orderId,
        };
    }

    getTokenAndLabelByOrderInfo(): ITokenPageQuery {
        const {
            offerInfoByToken: {offerInfo},
        } = this.props;
        const {offerOrderInfo} = offerInfo || {};
        const {token, label} = offerOrderInfo || {};

        return {
            token,
            label,
        };
    }

    getPermalinkFromOfferInfo(): number | undefined {
        const {
            offerInfoByToken: {tokenInfo, offerInfo},
        } = this.props;

        const offerInfoPermalink = offerInfo?.hotelInfo?.permalink;

        if (offerInfoPermalink) {
            return offerInfoPermalink;
        }

        const tokenInfoPermalink = tokenInfo?.permalink;

        if (tokenInfoPermalink) {
            return tokenInfoPermalink;
        }

        return undefined;
    }

    getTokenAndLabelByLocation(): ITokenPageQuery {
        const {location} = this.props;
        const {token, label} = getQueryByLocation(location) as ITokenPageQuery;

        return {
            token,
            label,
        };
    }

    reachLocationGoal(
        nextPagePath: string,
        key: EHotelsGoal,
        params: TMetrikaHotelsParams,
    ): void {
        const {
            location: {pathname},
        } = this.props;

        if (this.isSamePathname(pathname, nextPagePath)) {
            reachGoal(key, {hotels: params});
        }
    }

    getPathByOfferStatus(): string | undefined {
        const {
            orderInfo: {orderId},
            offerInfoByToken: {status: offerStatus, offerInfo},
        } = this.props;
        let nextPagePath = '';
        let pageBaseQuery: TPageQuery = {};
        let goalTarget;
        let goalParams;

        switch (offerStatus) {
            case OFFER_IS_NOT_AVAILABLE: {
                nextPagePath = URLs.hotelsBookError;
                pageBaseQuery = this.getTokenAndLabelByOrderInfo();
                goalTarget = EHotelsGoal.BOOK_PAGE_ERROR;
                goalParams = {offerError: 'offer_is_not_avaliable'};

                break;
            }

            case UNKNOWN_OFFER_ERROR: {
                nextPagePath = URLs.hotelsBookError;
                pageBaseQuery = this.getTokenAndLabelByLocation();
                goalTarget = EHotelsGoal.BOOK_PAGE_ERROR;
                goalParams = {offerError: '500'};

                break;
            }

            case OFFER_NOT_FOUND_ERROR: {
                const permalink = this.getPermalinkFromOfferInfo();

                if (permalink) {
                    nextPagePath = hotelsURLs.getHotelUrlByPermalink(permalink);
                } else {
                    nextPagePath = URLs.hotels;
                }

                goalTarget = EHotelsGoal.BOOK_PAGE_ERROR;
                goalParams = {offerError: 'offer_not_found'};

                break;
            }

            case OFFER_FETCHED: {
                nextPagePath = URLs.hotelsBook;

                if (orderId) {
                    pageBaseQuery = {orderId};
                } else if (offerInfo?.offerOrderInfo) {
                    const {
                        offerOrderInfo: {token, label},
                    } = offerInfo;

                    pageBaseQuery = {
                        token,
                        label,
                    };
                } else {
                    pageBaseQuery = this.getTokenAndLabelByLocation();
                }

                break;
            }

            case WRONG_OFFER_TOKEN: {
                goalTarget = EHotelsGoal.BOOK_PAGE_ERROR;
                goalParams = {offerError: 'wrong_offer_token'};

                return URLs.notFound;
            }
        }

        if (goalTarget) {
            this.reachLocationGoal(nextPagePath, goalTarget, goalParams);
        }

        return this.getHotelsPath(nextPagePath, pageBaseQuery);
    }

    isEmailInvalidCancelledStatus = (
        orderCancellationDetails: any | undefined,
    ): boolean => {
        const {reason} = orderCancellationDetails || {};

        return (
            reason === 'OrderCancellationDetailsReasonType.INVALID_INPUT' &&
            getInvalidInputDetailsFieldType(orderCancellationDetails) ===
                'OrderCancellationDetailsFieldType.EMAIL'
        );
    };

    /**
     * Возвращает url в зависимости от статуса заказа.
     * @param {boolean} withGoal Если true то с редиректы происходят с отправкой метрики.
     * @returns {string | undefined}
     */
    sendEventToMetrica(nextPagePath: string | undefined): void {
        const {
            orderInfo: {status: orderStatus, orderCancellationDetails},
        } = this.props;

        let goalTarget: EHotelsGoal | undefined;
        let goalParams;

        switch (orderStatus) {
            case CONFIRMED: {
                reachGoal(EHotelsGoal.HAPPY_PAGE_SHOWN);

                break;
            }

            case UNKNOWN_ORDER_ERROR: {
                goalTarget = EHotelsGoal.HAPPY_PAGE_ERROR;
                goalParams = {orderError: orderStatus.toLowerCase()};

                break;
            }

            case REFUNDED: {
                goalTarget = EHotelsGoal.ORDER_CANCELLED_PAGE_SHOWN;

                break;
            }

            case REFUND_FAILED: {
                goalTarget = EHotelsGoal.ORDER_CANCELLED_PAGE_ERROR;
                goalParams = {orderError: orderStatus.toLowerCase()};

                break;
            }

            case NEED_AUTHORIZATION: {
                goalTarget = EHotelsGoal.ORDER_AUTH_PAGE_SHOWN;

                break;
            }

            case CANCELLED: {
                if (
                    !this.isEmailInvalidCancelledStatus(
                        orderCancellationDetails,
                    )
                ) {
                    goalTarget = EHotelsGoal.HAPPY_PAGE_ERROR;
                    goalParams = {orderError: orderStatus.toLowerCase()};
                }

                break;
            }
            case CANCELLED_WITH_REFUND:
            case FAILED: {
                goalTarget = EHotelsGoal.HAPPY_PAGE_ERROR;
                goalParams = {orderError: orderStatus.toLowerCase()};

                break;
            }
        }

        if (nextPagePath && goalTarget) {
            this.reachLocationGoal(nextPagePath, goalTarget, goalParams);
        }
    }

    checkOfferStatus(
        prevProps: IBookRedirectControllerProps,
    ): string | undefined {
        const {
            offerInfoByToken: {status: prevOfferStatus},
        } = prevProps;
        const {
            offerInfoByToken: {status: offerStatus},
        } = this.props;

        if (prevOfferStatus !== offerStatus) {
            return this.getPathByOfferStatus();
        }

        return undefined;
    }

    checkOrderStatus(
        prevProps: IBookRedirectControllerProps,
    ): string | undefined {
        const {
            orderInfo: {status: prevOrderStatus},
        } = prevProps;
        const {
            orderInfo: {orderId, status: orderStatus, orderCancellationDetails},
            location,
        } = this.props;

        if (prevOrderStatus !== orderStatus) {
            const path = hotelsURLs.getPathByOrderStatus({
                orderId,
                orderStatus,
                orderCancellationDetails,
                location,
            });

            this.sendEventToMetrica(path);

            return path;
        }

        return undefined;
    }

    render(): ReactNode {
        return null;
    }
}

export default bookInfoProvider()(BookRedirectController);
