import {PureComponent} from 'react';
import {connect} from 'react-redux';
import _flow from 'lodash/flow';
import _isEqual from 'lodash/isEqual';
import _pick from 'lodash/pick';
import {withRouter, RouteComponentProps} from 'react-router-dom';

import {IHotelSlugRouteMatch} from 'types/hotels/hotel/IHotelSlugRouteMatch';

import {StoreInterface} from 'reducers/storeTypes';
import {
    fetchAndSelectSearchSuggestAction,
    FetchAndSelectSearchSuggestActionType,
    resetSessionInfoSuggestAction,
    TResetSessionInfoSuggestAction,
} from 'reducers/depreacted/hotels/searchSuggests/actions';
import {
    updateSearchInformation as updateSearchInformationAction,
    UpdateSearchInformationActionType,
    clearSearchInformation as clearSearchInformationAction,
    ClearSearchInformationActionType,
} from 'reducers/depreacted/hotels/searchInformation/actions';

/* Selectors */
import {
    getSearchInformation,
    GetSearchInformationType,
} from 'selectors/depreacted/hotels/search/searchInformation/searchInformationSelector';

/* Utilities */
import {getHotelSlugByRouteMatch} from 'projects/depreacted/hotels/utilities/getHotelIdentifier/getHotelIdentifier';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {validateSearchHotelsParams} from 'projects/depreacted/hotels/utilities/validateSearchHotelsParams/validateSearchHotelsParams';
import {getRegionSlugByRouteMatch} from 'projects/depreacted/hotels/utilities/getRegionIdentifier/getRegionIdentifier';

/* React-Redux Types */
interface IHotelsSearchInformationProviderStateProps {
    searchInformation: ReturnType<GetSearchInformationType>;
}

interface IHotelsSearchInformationProviderDispatchProps {
    fetchAndSelectSearchSuggest: FetchAndSelectSearchSuggestActionType;
    updateSearchInformation: UpdateSearchInformationActionType;
    resetSessionInfoSuggest: TResetSessionInfoSuggestAction;
    clearSearchInformation: ClearSearchInformationActionType;
}

/* React-Redux map helpers */
const mapStateToProps = (
    state: StoreInterface,
): IHotelsSearchInformationProviderStateProps => ({
    searchInformation: getSearchInformation(state),
});

const mapDispatchToProps = {
    fetchAndSelectSearchSuggest: fetchAndSelectSearchSuggestAction,
    updateSearchInformation: updateSearchInformationAction,
    resetSessionInfoSuggest: resetSessionInfoSuggestAction,
    clearSearchInformation: clearSearchInformationAction,
};

export type HotelsSearchInformationProviderProps =
    IHotelsSearchInformationProviderStateProps &
        IHotelsSearchInformationProviderDispatchProps &
        RouteComponentProps<IHotelSlugRouteMatch>;

/* Constants */
const DATES_AND_GUESTS_PARAMS = [
    'startDate',
    'endDate',
    'checkinDate',
    'checkoutDate',
    'adults',
    'childrenAges',
];

class HotelsSearchInformationProvider extends PureComponent<HotelsSearchInformationProviderProps> {
    componentDidMount(): void {
        this.updateFormValuesAndSearchInformation();
    }

    componentDidUpdate(prevProps: HotelsSearchInformationProviderProps): void {
        this.checkUpdateLocation(prevProps);
    }

    componentWillUnmount(): void {
        const {resetSessionInfoSuggest} = this.props;

        resetSessionInfoSuggest();
    }

    private checkUpdateLocation(
        prevProps: HotelsSearchInformationProviderProps,
    ) {
        const {location} = this.props;

        if (location !== prevProps.location) {
            this.updateFormValuesAndSearchInformation();
        }
    }

    /* Actions */

    private updateFormValuesAndSearchInformation() {
        this.updateDatesAndGuestsParams();
        this.fillSearchSuggestByGeoIdOrPermalink();
    }

    /* Helpers */

    private getDatesAndGuestsSearchParamsByLocation = () => {
        const {location} = this.props;
        const {checkinDate, checkoutDate, childrenAges, adults} = _flow(
            getQueryByLocation,
            validateSearchHotelsParams,
        )(location, false);

        if (checkinDate && checkoutDate && adults) {
            return {
                startDate: checkinDate,
                endDate: checkoutDate,
                checkinDate,
                checkoutDate,
                childrenAges,
                adults,
            };
        }
    };

    /*
     * Update dates and guests params for reducers/hotels/searchInformation if need
     * */
    private updateDatesAndGuestsParams() {
        const {updateSearchInformation, searchInformation} = this.props;
        const datesAndGuestsSearchParams =
            this.getDatesAndGuestsSearchParamsByLocation();
        const searchInformationDateAndGuestsParams = _pick(
            searchInformation,
            DATES_AND_GUESTS_PARAMS,
        );

        if (
            datesAndGuestsSearchParams &&
            !_isEqual(
                datesAndGuestsSearchParams,
                searchInformationDateAndGuestsParams,
            )
        ) {
            updateSearchInformation(datesAndGuestsSearchParams);
        }
    }

    /*
     * Fetch suggest items and select first item by geoId or permalink if need
     * */
    private fillSearchSuggestByGeoIdOrPermalink() {
        const {
            match,
            location,
            fetchAndSelectSearchSuggest,
            clearSearchInformation,
            searchInformation: {geoObject, hotel},
        } = this.props;

        const {geoId, hotelPermalink} = _flow(
            getQueryByLocation,
            validateSearchHotelsParams,
        )(location, false);
        const hotelSlug = getHotelSlugByRouteMatch(match);
        const regionSlug = getRegionSlugByRouteMatch(match);

        switch (true) {
            case Boolean(geoId): {
                if (geoId !== Number(geoObject.geoId)) {
                    fetchAndSelectSearchSuggest({geoId});
                }

                break;
            }

            case Boolean(hotelSlug): {
                if (hotelSlug !== hotel.hotelSlug) {
                    fetchAndSelectSearchSuggest({hotelSlug});
                }

                break;
            }

            case Boolean(hotelPermalink): {
                if (hotelPermalink !== hotel.permalink) {
                    fetchAndSelectSearchSuggest({permalink: hotelPermalink});
                }

                break;
            }

            case Boolean(regionSlug): {
                if (regionSlug !== geoObject.regionSlug) {
                    fetchAndSelectSearchSuggest({regionSlug});
                }

                break;
            }

            default: {
                clearSearchInformation();
            }
        }
    }

    render() {
        return null;
    }
}

export default _flow(
    withRouter,
    connect<
        IHotelsSearchInformationProviderStateProps,
        IHotelsSearchInformationProviderDispatchProps,
        {},
        StoreInterface
    >(mapStateToProps, mapDispatchToProps),
)(HotelsSearchInformationProvider);
