import {PureComponent, ReactNode} from 'react';
import _times from 'lodash/times';

import {IWithDeviceType} from 'types/withDeviceType';
import {EFormGroup} from 'types/hotels/book/IBookFormValues';

import {
    prepareQaAttributes,
    IWithQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-BookGuestsForm';

import Form from 'components/Form/Form';
import Separator from 'components/Separator/Separator';
import Intersperse from 'components/Intersperse/Intersperse';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookingContactsForm from 'components/BookingContactsForm/BookingContactsForm';

import {TBookPageContainerProps} from './BookGuestsFormContainer';

import cx from './BookGuestsForm.scss';

const BOOK_FORM_QA = 'book-form';

export interface IBookGuestsFormOwnProps
    extends IWithDeviceType,
        IWithQaAttributes {
    searchParams?: any;
}

interface IBookGuestsFormProps
    extends TBookPageContainerProps,
        IBookGuestsFormOwnProps {}

class BookGuestsForm extends PureComponent<IBookGuestsFormProps> {
    /* Render field */

    private renderGuestForm(title: string, index: number): ReactNode {
        const {searchParams} = this.props;
        const {adults = 0, childrenAges = []} = searchParams || {};
        const totalPassengersLength = adults + childrenAges.length;

        return (
            <div
                className={cx('passengerForm')}
                style={{zIndex: totalPassengersLength - index + 1}}
            ></div>
        );
    }

    /* Render fields */

    private renderGuestsFields(): ReactNode {
        const {searchParams} = this.props;
        const {adults = 0, childrenAges = []} = searchParams || {};

        return (
            <Intersperse separator={<Separator />}>
                {_times(adults, index => (
                    <Form.FieldGroup groupId={EFormGroup.adult} key={index}>
                        <Form.FieldGroup groupId={index}>
                            <BookingLayout.Card.Section>
                                {this.renderGuestForm(i18nBlock.adult(), index)}
                            </BookingLayout.Card.Section>
                        </Form.FieldGroup>
                    </Form.FieldGroup>
                ))}

                {childrenAges.map((age, index) => (
                    <Form.FieldGroup key={index} groupId={EFormGroup.children}>
                        <Form.FieldGroup groupId={index}>
                            <BookingLayout.Card.Section>
                                {this.renderGuestForm(
                                    i18nBlock.child({age}),
                                    adults + index,
                                )}
                            </BookingLayout.Card.Section>
                        </Form.FieldGroup>
                    </Form.FieldGroup>
                ))}
            </Intersperse>
        );
    }

    renderEmailAndPhoneFields(): ReactNode {
        return (
            <Form.FieldGroup groupId={EFormGroup.contacts}>
                <BookingContactsForm
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: `${BOOK_FORM_QA}-contacts`,
                    })}
                />
            </Form.FieldGroup>
        );
    }

    render(): ReactNode {
        return (
            <div
                {...prepareQaAttributes({
                    parent: this.props,
                    current: BOOK_FORM_QA,
                })}
            >
                <BookingLayout.Forms>
                    <BookingLayout.Card sectioned>
                        {this.renderGuestsFields()}
                    </BookingLayout.Card>
                    <BookingLayout.Card className={cx('contacts')}>
                        {this.renderEmailAndPhoneFields()}
                    </BookingLayout.Card>
                </BookingLayout.Forms>
            </div>
        );
    }
}

export default BookGuestsForm;
