import {createRef, PureComponent, ReactNode} from 'react';
import {FormSpy} from 'react-final-form';
import {FormState} from 'final-form';
import _isEqual from 'lodash/isEqual';

import {EFormKey} from 'constants/form/EFormKey';
import {
    BOOK_FORM_SUBMITTED,
    CANCELLED,
    IN_PROGRESS,
    PAYMENT_FAILED,
    RESERVED,
    RESERVED_WITH_RESTRICTIONS,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';

import {IWithDeviceType} from 'types/withDeviceType';
import {
    EFormGroup,
    IBookFormValues,
    IStorageBookGuest,
} from 'types/hotels/book/IBookFormValues';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';
import {IPromoCodeApplicationResult} from 'types/hotels/book/IApplyPromoCodes';
import {IBookUserSelect} from 'types/hotels/book/ICreateOrder';

import {IOrderInfoReducer} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/reducer';
import {TUserInfo} from 'reducers/common/commonReducerTypes';

import {getReason} from 'selectors/depreacted/hotels/book/orderCancellationDetails/getReason';
import {getInvalidInputDetailsFieldType} from 'selectors/depreacted/hotels/book/orderCancellationDetails/getInvalidInputDetailsFieldType';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {saveBookingPassengers} from 'utilities/localStorage/booking/saveBookingPassengers';
import {saveBookingContacts} from 'utilities/localStorage/booking/saveBookingContacts';
import getFinishPaymentPagePath from 'projects/depreacted/hotels/utilities/getFinishPaymentPagePath/getFinishPaymentPagePath';
import getTotalTopupPlusPoints from 'projects/depreacted/hotels/pages/BookPage/utilities/getTotalTopupPlusPoints';
import getTotalWithdrawPlusPoints from 'projects/depreacted/hotels/pages/BookPage/utilities/getTotalWithdrawPlusPoints';

import * as i18nBlock from 'i18n/hotels-BookPayPage';

import {MODAL_ANIMATION_DURATION_MS} from 'components/Modal/Modal';
import YandexEdaPromo from 'components/YandexEdaPromo/YandexEdaPromo';
import BookingLayout from 'components/Layouts/BookingLayout/BookingLayout';
import BookInvalidInputModal from 'components/BookInvalidInputModal/BookInvalidInputModal';
import BookLicenseAgreement from 'projects/depreacted/hotels/components/BookLicenseAgreement/BookLicenseAgreement';
import BookPriceInfo from 'projects/depreacted/hotels/components/BookPriceInfo/BookPriceInfo';
import BookLoader from 'components/BookLoader/BookLoader';
import OfferHotelInfo from 'projects/depreacted/hotels/components/OfferHotelInfo/OfferHotelInfo';
import GoToForm from 'projects/depreacted/hotels/components/GoToForm/GoToForm';
import MirCashbackCart from './components/MirCashbackCart/MirCashbackCart';
import BookSupportPhone from './components/BookSupportPhone/BookSupportPhone';
import MirCashbackDisclaimer from './components/MirCashbackDisclaimer/MirCashbackDisclaimer';
import ForceLogin from 'projects/depreacted/hotels/components/ForceLogin/ForceLogin';
import CovidInfo from 'projects/depreacted/hotels/components/CovidInfo/CovidInfo';

import BookGuestsForm from '../BookGuestsForm/BookGuestsFormContainer';
import {TBookPageContentContainerProps} from './BookPageContentContainer';
import {
    TChangeFieldEvent,
    TFocusFieldEvent,
    TFormEvent,
} from '../FormController/FormController';
import ReservedWithRestrictionsModal from '../ReservedWithRestrictionsModal/ReservedWithRestrictionsModal';

import cx from './BookPageContent.scss';

const EMAIL_INPUT_FOCUS_TIMEOUT = MODAL_ANIMATION_DURATION_MS + 10;

const orderNotReadyStatuses = [RESERVED, IN_PROGRESS, BOOK_FORM_SUBMITTED];

const _noop = (): void => {};

export interface IBookPageContentOwnProps
    extends IWithDeviceType,
        IWithQaAttributes {
    bookUserSelect: IBookUserSelect;
    offerInfo: any;
    orderInfo: IOrderInfoReducer;
    userInfo: TUserInfo;

    canRenderGoToForm: boolean;
    wasSessionKeyRenewed: boolean;

    tokenByLocation: string;
    focusFirstField: TFormEvent;
    focusFieldByName: TFocusFieldEvent;
    submitForm: TFormEvent;
    formState: IBookFormValues;
    changeFieldByName: TChangeFieldEvent;
}

interface IBookPageContentProps
    extends IBookPageContentOwnProps,
        TBookPageContentContainerProps {}

interface IBookPageContentState {
    canRenderGoToForm: boolean;
    // Флаг, что "можно" показать модалку "Невалидный емейл".
    isEmailInvalidOnPartner: boolean;
    isPaymentFailedModalClosed: boolean;
    isReservedWithRestriptionModalClosed: boolean;
}

class BookPageContent extends PureComponent<
    IBookPageContentProps,
    IBookPageContentState
> {
    bookPriceInfoRef = createRef<HTMLDivElement>();

    state = {
        canRenderGoToForm: true,
        isEmailInvalidOnPartner: false,
        isPaymentFailedModalClosed: false,
        isReservedWithRestriptionModalClosed: false,
    };

    componentDidMount(): void {
        this.checkRenderGoToForm();
        this.checkIsEmailInvalidOnPartner();
    }

    componentDidUpdate(): void {
        /**
         * эта синхроназация со стором нужна в первую очередь, чтобы не сломать:
         * src/redux/sagas/hotels/bookAndPay/pillingOrder.ts:143 – ставится другая кровать при фейле
         * src/redux/sagas/hotels/bookAndPay/estimateDiscount.ts:25 – сбрасывается кровать и рассрочка,
         * если она не доступна при выбранной скидке
         */
        const {useDeferredPayments, bedsGroupId} = this.props.bookUserSelect;

        if (
            this.props.formState[EFormGroup.priceInfo]?.useDeferredPayments !==
            useDeferredPayments
        ) {
            this.props.changeFieldByName(
                'priceInfo.useDeferredPayments',
                useDeferredPayments,
            );
        }

        if (
            this.props.formState[EFormGroup.priceInfo]?.bedsGroupId !==
            bedsGroupId
        ) {
            this.props.changeFieldByName('priceInfo.bedsGroupId', bedsGroupId);
        }
    }

    private checkRenderGoToForm(): void {
        const {
            orderInfo: {status},
        } = this.props;
        const {canRenderGoToForm} = this.state;

        if (status === PAYMENT_FAILED && canRenderGoToForm) {
            this.setState({canRenderGoToForm: false});
        }
    }

    private checkIsEmailInvalidOnPartner(): void {
        const {
            orderInfo: {status, orderCancellationDetails},
        } = this.props;

        if (status === CANCELLED) {
            const isEmailInvalid =
                getReason(orderCancellationDetails) ===
                    'OrderCancellationDetailsReasonType.INVALID_INPUT' &&
                getInvalidInputDetailsFieldType(orderCancellationDetails) ===
                    'OrderCancellationDetailsFieldType.EMAIL';

            if (isEmailInvalid) {
                this.setState({isEmailInvalidOnPartner: true});
            }
        }
    }

    private fetchEstimatedDiscount(
        plusMode: EYandexPlusApplicationMode,
        promoCodes: string[],
    ): void {
        const {
            bookUserSelect,
            offerInfo: {offerOrderInfo},
            estimateDiscount,
            promoCodesInfo,
        } = this.props;

        const isPlusWithdraw = plusMode === EYandexPlusApplicationMode.WITHDRAW;
        const isDeferredPaymentNotAvailable =
            isPlusWithdraw || !promoCodesInfo?.data?.deferredPaymentSchedule;

        const finalPoints = isPlusWithdraw
            ? this.getTotalWithdrawPlusPoints()
            : this.getTotalTopupPlusPoints();

        estimateDiscount({
            bookUserSelect: {
                ...bookUserSelect,
                useDeferredPayments: isDeferredPaymentNotAvailable
                    ? false
                    : bookUserSelect.useDeferredPayments,
            },
            offerOrderInfo,
            promoCodes,
            appliedPromoCampaigns: finalPoints
                ? {
                      yandexPlus: {
                          points: finalPoints,
                          mode: plusMode,
                      },
                  }
                : undefined,
        });
    }

    /* Helpers */

    private getTotalWithdrawPlusPoints = (): number | undefined => {
        const {promoCodesInfo, offerInfo} = this.props;

        return getTotalWithdrawPlusPoints({
            promoCampaigns: offerInfo?.promoCampaigns,
            promoCodesInfo,
        });
    };

    private getTotalTopupPlusPoints = (): number => {
        const {promoCodesInfo, offerInfo} = this.props;

        return getTotalTopupPlusPoints({
            promoCampaigns: offerInfo?.promoCampaigns,
            promoCodesInfo,
        });
    };

    private getBorderTopToHidden = (): number | undefined => {
        const bookPriceInfoNode = this.bookPriceInfoRef.current;

        if (!bookPriceInfoNode) {
            return undefined;
        }

        return (
            bookPriceInfoNode.getBoundingClientRect().top + window.pageYOffset
        );
    };

    private isMirCashbackEligible(): boolean {
        const {
            promoCodesInfo,
            offerInfo: {promoCampaigns},
        } = this.props;

        return promoCodesInfo.data?.promoCampaigns.mir2020?.eligible ===
            undefined
            ? Boolean(promoCampaigns?.mir2020?.eligible)
            : promoCodesInfo.data.promoCampaigns.mir2020.eligible;
    }

    private getAppliedPromoCode(): IPromoCodeApplicationResult | undefined {
        const {
            promoCodesInfo,
            orderInfo: {orderPriceInfo, status},
        } = this.props;

        const codeApplicationResults =
            orderPriceInfo?.codeApplicationResults || [];

        if (promoCodesInfo?.data?.codeApplicationResults[0]) {
            return promoCodesInfo.data.codeApplicationResults[0];
        }

        if (status !== RESERVED_WITH_RESTRICTIONS) {
            return codeApplicationResults[0];
        }
    }

    /* Handlers */

    private handleGoToForm = (): void => {
        const {focusFirstField} = this.props;

        focusFirstField();
        this.setState({canRenderGoToForm: false});
    };

    private handleFailPaymentModalAction = (): void => {
        const {focusFieldByName} = this.props;

        this.setState({isEmailInvalidOnPartner: false}, () => {
            // Триггерим фокус на email инпут
            setTimeout(
                () => focusFieldByName('contacts.email'),
                EMAIL_INPUT_FOCUS_TIMEOUT, // Нужен таймаут, тк у модалки есть анимация закрытия
            );
        });
    };

    private handleFormStateChange = (
        formState: FormState<IBookFormValues>,
    ): void => {
        const {userInfo, setBookUserSelect} = this.props;
        const {submitSucceeded, values} = formState;
        const {adult = [], children = [], priceInfo} = values;
        const adultGuests: IStorageBookGuest[] = adult.map(guest => ({
            ...guest,
            ageGroup: 'adult',
        }));
        const childrenGuests: IStorageBookGuest[] = children.map(guest => ({
            ...guest,
            ageGroup: 'children',
        }));

        if (priceInfo && !_isEqual(priceInfo, this.props.bookUserSelect)) {
            setBookUserSelect(priceInfo);
        }

        if (
            priceInfo &&
            priceInfo.plusMode !==
                this.props.formState[EFormGroup.priceInfo]?.plusMode
        ) {
            this.handlePlusModeChange(priceInfo.plusMode);
        }

        if (!submitSucceeded) {
            saveBookingPassengers<IStorageBookGuest>(
                EFormKey.HOTEL_BOOK,
                [...adultGuests, ...childrenGuests],
                userInfo,
            );

            if (values.contacts) {
                saveBookingContacts(
                    EFormKey.HOTEL_BOOK,
                    values.contacts,
                    userInfo,
                );
            }
        }
    };

    private handlePlusModeChange = (
        plusMode: EYandexPlusApplicationMode,
    ): void => {
        const promoCode = this.getAppliedPromoCode()?.code;

        this.fetchEstimatedDiscount(plusMode, promoCode ? [promoCode] : []);
    };

    private handleApplyPromoCodes = (promoCodes: string[]): void => {
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            this.props.formState[EFormGroup.priceInfo] || {};

        this.fetchEstimatedDiscount(plusMode, promoCodes);
    };

    private handleResetPromoCodes = (): void => {
        const {resetPromoCodes, formState} = this.props;
        const {plusMode = EYandexPlusApplicationMode.TOPUP} =
            formState[EFormGroup.priceInfo] || {};

        resetPromoCodes();
        this.fetchEstimatedDiscount(plusMode, []);
    };

    private handleContinueOrder = (): void => {
        const {orderInfo, startOrderPolling} = this.props;

        if (
            orderInfo &&
            orderInfo.orderId &&
            orderInfo.status === RESERVED_WITH_RESTRICTIONS
        ) {
            const finishPaymentPagePath = getFinishPaymentPagePath();

            startOrderPolling({
                orderId: orderInfo.orderId,
                finishPaymentPagePath,
                withRestrictions: true,
            });
        }
    };

    private handleCancelOrder = (): void => {
        const {
            removeOrder,
            offerInfo,
            orderInfo: {orderId},
        } = this.props;

        if (orderId && offerInfo) {
            const {
                offerOrderInfo: {token, label},
            } = offerInfo;

            this.setState({isReservedWithRestriptionModalClosed: true}, () => {
                removeOrder({orderId, token, label});
            });
        }
    };

    /* Render */

    private renderOrderInfoLoader(): ReactNode {
        const {
            orderInfo: {status},
            offerInfo: {deferredPaymentSchedule},
            bookUserSelect,
        } = this.props;

        const canRenderLoader = orderNotReadyStatuses.includes(status);
        const noPaymentStep =
            bookUserSelect.useDeferredPayments &&
            deferredPaymentSchedule?.zeroFirstPayment;

        return (
            <BookLoader
                title={i18nBlock.createOrderDotTitle()}
                description={
                    noPaymentStep ? '' : i18nBlock.createOrderDotDescription()
                }
                isLoading={canRenderLoader}
                isModalView
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'createOrderLoader',
                })}
            />
        );
    }

    /* left column */

    private renderHotelInfoWithBadge(): ReactNode {
        const {offerInfo, deviceType} = this.props;

        if (!offerInfo) {
            return null;
        }

        return (
            <>
                {deviceType.isMobile && (
                    <BookSupportPhone className={cx('supportPhone')} />
                )}
                {this.renderHotelInfo()}
            </>
        );
    }

    private renderYandexEdaPromo(): ReactNode {
        const {
            offerInfo: {promoCampaigns},
        } = this.props;

        if (!promoCampaigns?.yandexEda?.eligible) {
            return null;
        }

        return (
            <BookingLayout.Card>
                <YandexEdaPromo data={promoCampaigns.yandexEda.data} />
            </BookingLayout.Card>
        );
    }

    private renderHotelInfo(): ReactNode {
        const {offerInfo, bookUserSelect, deviceType} = this.props;

        return (
            <BookingLayout.Card className={cx('hotelInfo')}>
                <OfferHotelInfo
                    partnerId={offerInfo.partnerId}
                    roomInfo={offerInfo.roomInfo}
                    hotelInfo={offerInfo.hotelInfo}
                    mealInfo={offerInfo.mealInfo}
                    deviceType={deviceType}
                    searchParams={offerInfo.searchParams}
                    bedsGroups={offerInfo.bedsGroups}
                    partnerHotelInfo={offerInfo.partnerHotelInfo}
                    setBedsGroupId={_noop}
                    cancellationInfo={offerInfo.refundableInfo}
                    bookUserSelect={bookUserSelect}
                    {...prepareQaAttributes(this.props)}
                />
            </BookingLayout.Card>
        );
    }

    private renderGuestsForm(): ReactNode {
        const {offerInfo, deviceType} = this.props;

        return (
            offerInfo && (
                <BookGuestsForm
                    deviceType={deviceType}
                    searchParams={offerInfo.searchParams}
                    {...prepareQaAttributes(this.props)}
                />
            )
        );
    }

    private renderLeftColumn(): ReactNode {
        const {deviceType} = this.props;

        return (
            <>
                <BookingLayout.Snippets>
                    <CovidInfo
                        deviceType={deviceType}
                        className={cx('banner')}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'hotel-covid-info',
                        })}
                    />
                    {this.renderHotelInfoWithBadge()}
                    {this.renderYandexEdaPromo()}
                </BookingLayout.Snippets>

                {this.renderGuestsForm()}
                {deviceType.isDesktop && this.renderMirDisclaimer()}
            </>
        );
    }

    /* right content */

    private renderPriceInfo(): ReactNode {
        const {
            offerInfo,
            orderInfo: {guestsInfo, status},
            deviceType,
            userInfo,
            promoCodesInfo,
            formState,
            experiments,
        } = this.props;

        return (
            offerInfo && (
                <BookPriceInfo
                    userInfo={userInfo}
                    priceInfo={offerInfo.priceInfo}
                    searchParams={offerInfo.searchParams}
                    promoCampaigns={offerInfo.promoCampaigns}
                    deviceType={deviceType}
                    guestsInfo={guestsInfo}
                    deferredPaymentSchedule={offerInfo.deferredPaymentSchedule}
                    priceAfterPlusWithdrawBottom={
                        experiments.priceAfterPlusWithdrawBottom
                    }
                    cancellationInfo={offerInfo.refundableInfo}
                    forwardedRef={this.bookPriceInfoRef}
                    promoCodesInfo={promoCodesInfo}
                    formState={formState}
                    appliedPromoCode={this.getAppliedPromoCode()}
                    orderStatus={status}
                    onApplyPromoCodesClick={this.handleApplyPromoCodes}
                    onResetPromoCodesClick={this.handleResetPromoCodes}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'priceInfo',
                    })}
                />
            )
        );
    }

    private renderLicenseAgreements(): ReactNode {
        const {
            offerInfo,
            deviceType: {isMobile},
        } = this.props;

        return (
            offerInfo && (
                <BookLicenseAgreement
                    isMobile={isMobile}
                    legalInfo={offerInfo.legalInfo}
                />
            )
        );
    }

    private renderRightColumn(): ReactNode {
        const {
            offerInfo: {promoCampaigns},
            promoCodesInfo,
        } = this.props;

        const mirCashbackIsEligible = this.isMirCashbackEligible();

        const mirCashbackString =
            promoCodesInfo.data?.promoCampaigns.mir2020?.eligible === undefined
                ? promoCampaigns?.mir2020?.cashbackAmountString
                : promoCodesInfo.data.promoCampaigns.mir2020
                      .cashbackAmountString;

        return (
            <>
                <ForceLogin className={cx('forceLogin')} isBookPage />
                <BookingLayout.Cart className={cx('cart')}>
                    {this.renderPriceInfo()}
                </BookingLayout.Cart>

                {mirCashbackIsEligible && mirCashbackString && (
                    <MirCashbackCart
                        className={cx('mirCashback')}
                        cashback={mirCashbackString}
                    />
                )}

                <BookingLayout.CartCaption>
                    {this.renderLicenseAgreements()}
                </BookingLayout.CartCaption>
            </>
        );
    }

    /* Modals */

    private renderGoToForm(): ReactNode {
        const {canRenderGoToForm} = this.state;
        const {offerInfo} = this.props;
        const totalPrice = offerInfo?.priceInfo.hotelCharges.totals?.totalPrice;

        const mirCashbackIsEligible = this.isMirCashbackEligible();

        return (
            totalPrice && (
                <GoToForm
                    price={totalPrice}
                    active={canRenderGoToForm}
                    getBorderTopToHidden={this.getBorderTopToHidden}
                    mirCashbackIsEligible={mirCashbackIsEligible}
                    onButtonClick={this.handleGoToForm}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'goToForm',
                    })}
                />
            )
        );
    }

    private renderInvalidEmailModal(): ReactNode {
        const {isEmailInvalidOnPartner} = this.state;
        const {orderInfo, wasSessionKeyRenewed} = this.props;
        const customerEmail = orderInfo.guestsInfo?.customerEmail;

        return (
            <BookInvalidInputModal
                actionHandler={this.handleFailPaymentModalAction}
                customerEmail={customerEmail}
                fieldType={'OrderCancellationDetailsFieldType.EMAIL'}
                // Хотим увидеть модалку только после обновления sessionKey
                isVisible={isEmailInvalidOnPartner && wasSessionKeyRenewed}
            />
        );
    }

    private renderReservedWithRestrictionsModal(): ReactNode {
        const {isReservedWithRestriptionModalClosed} = this.state;
        const {
            orderInfo: {status, orderPriceInfo},
            promoCodesInfo: {data},
        } = this.props;

        return (
            <ReservedWithRestrictionsModal
                promoCodesInfo={data}
                codeApplicationResults={
                    orderPriceInfo?.codeApplicationResults[0]
                }
                isVisible={
                    status === RESERVED_WITH_RESTRICTIONS &&
                    !isReservedWithRestriptionModalClosed
                }
                onContinueClick={this.handleContinueOrder}
                onCancelClick={this.handleCancelOrder}
            />
        );
    }

    private renderFormSpy(): ReactNode {
        return (
            <FormSpy<IBookFormValues>
                onChange={this.handleFormStateChange}
                subscription={{
                    values: true,
                    submitSucceeded: true,
                }}
            />
        );
    }

    private renderMirDisclaimer(): ReactNode {
        const {deviceType} = this.props;

        if (!this.isMirCashbackEligible()) {
            return null;
        }

        return (
            <MirCashbackDisclaimer
                className={cx(
                    'disclaimer',
                    deviceModMobile('disclaimer', deviceType),
                )}
            />
        );
    }

    render(): ReactNode {
        const {deviceType, canRenderGoToForm} = this.props;

        return (
            <>
                <BookingLayout
                    className={cx('root', deviceMods('root', deviceType))}
                    deviceType={deviceType}
                    leftColumn={this.renderLeftColumn()}
                    rightColumn={this.renderRightColumn()}
                />

                {deviceType.isMobile && this.renderMirDisclaimer()}

                {this.renderOrderInfoLoader()}
                {deviceType.isMobile &&
                    canRenderGoToForm &&
                    this.renderGoToForm()}
                {this.renderInvalidEmailModal()}
                {this.renderFormSpy()}
                {this.renderReservedWithRestrictionsModal()}
            </>
        );
    }
}

export default BookPageContent;
