import {EFormKey} from 'constants/form/EFormKey';
import {
    FREE_PHONE_REGEXP,
    getEmailValidationRegExp,
} from 'constants/form/validationRules';
import {EFieldName as EContactsFieldName} from 'components/BookingContactsForm/constants/fieldNames';

import {
    IFormValidationInfo,
    IFieldValidation,
} from 'types/common/validation/form';
import {
    EValidationType,
    IDependentValidation,
} from 'types/common/validation/validation';
import {
    EBookGuestFieldName,
    EFormGroup,
} from 'types/hotels/book/IBookFormValues';

import * as i18nBookGuestsFormBlock from 'i18n/hotels-BookGuestsForm';

const NAME_MAX_LENGTH = 80;
const EMAIL_MAX_LENGTH = 40;
const PHONE_MAX_LENGTH = 80;

const nameValidation: IFieldValidation = {
    blur: [
        {
            type: EValidationType.regex,
            params: '^[a-zA-Zа-яА-ЯёЁ-\\s]*$',
            errorMessage:
                i18nBookGuestsFormBlock.validateDotGuestNameExtended(),
        },
        {
            type: EValidationType.maxLength,
            params: NAME_MAX_LENGTH,
            errorMessage: i18nBookGuestsFormBlock.validateDotMaxLengthExceeded({
                maxLength: NAME_MAX_LENGTH,
            }),
        },
    ],
};

const firstAdultValidation: IDependentValidation = {
    conditions: [
        {
            path: {
                fieldName: EBookGuestFieldName.index,
            },
            value: [
                {
                    type: EValidationType.oneOf,
                    params: [0],
                },
            ],
        },
    ],
    validation: {
        submit: [
            {
                type: EValidationType.required,
                params: true,
                errorMessage:
                    i18nBookGuestsFormBlock.validateDotRequiredField(),
            },
        ],
    },
};

// Если заполнена фамилия то нужно заполнить и имя
const optionalFirstNameValidation: IDependentValidation = {
    conditions: [
        {
            path: {
                fieldName: 'EFieldName.lastName',
            },
            value: [
                {
                    type: EValidationType.required,
                    params: true,
                },
            ],
        },
    ],
    validation: {
        submit: [
            {
                type: EValidationType.required,
                params: true,
                errorMessage:
                    i18nBookGuestsFormBlock.validateDotRequiredField(),
            },
        ],
    },
};

// Если заполнено имя то нужно заполнить и фамилию
const optionalLastNameValidation: IDependentValidation = {
    conditions: [
        {
            path: {
                fieldName: 'EFieldName.firstName',
            },
            value: [
                {
                    type: EValidationType.required,
                    params: true,
                },
            ],
        },
    ],
    validation: {
        submit: [
            {
                type: EValidationType.required,
                params: true,
                errorMessage:
                    i18nBookGuestsFormBlock.validateDotRequiredField(),
            },
        ],
    },
};

const validationInfo: IFormValidationInfo = {
    id: EFormKey.HOTEL_BOOK,

    fieldGroups: [
        {
            id: EFormGroup.adult,
            fields: [
                {
                    name: 'EFieldName.firstName',
                    validation: nameValidation,
                    dependentValidations: [
                        firstAdultValidation,
                        optionalFirstNameValidation,
                    ],
                },
                {
                    name: 'EFieldName.lastName',
                    validation: nameValidation,
                    dependentValidations: [
                        firstAdultValidation,
                        optionalLastNameValidation,
                    ],
                },
            ],
        },
        {
            id: EFormGroup.children,
            fields: [
                {
                    name: 'EFieldName.firstName',
                    validation: nameValidation,
                    dependentValidations: [optionalFirstNameValidation],
                },
                {
                    name: 'EFieldName.lastName',
                    validation: nameValidation,
                    dependentValidations: [optionalLastNameValidation],
                },
            ],
        },
        {
            id: EFormGroup.contacts,
            fields: [
                {
                    name: EContactsFieldName.email,
                    validation: {
                        blur: [
                            {
                                type: EValidationType.regex,
                                params: getEmailValidationRegExp({
                                    allowRussianCharacters: true,
                                }),
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotEmail(),
                            },
                            {
                                type: EValidationType.maxLength,
                                params: EMAIL_MAX_LENGTH,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotMaxLengthExceeded(
                                        {maxLength: EMAIL_MAX_LENGTH},
                                    ),
                            },
                        ],
                        submit: [
                            {
                                type: EValidationType.required,
                                params: true,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotRequiredField(),
                            },
                        ],
                    },
                },
                {
                    name: EContactsFieldName.phone,
                    validation: {
                        blur: [
                            {
                                type: EValidationType.regex,
                                params: FREE_PHONE_REGEXP,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotPhoneFormat(),
                            },
                            {
                                type: EValidationType.maxLength,
                                params: PHONE_MAX_LENGTH,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotMaxLengthExceeded(
                                        {maxLength: PHONE_MAX_LENGTH},
                                    ),
                            },
                        ],
                        submit: [
                            {
                                type: EValidationType.required,
                                params: true,
                                errorMessage:
                                    i18nBookGuestsFormBlock.validateDotRequiredField(),
                            },
                        ],
                    },
                },
            ],
        },
    ],
};

export default validationInfo;
