import React, {Component, ReactNode} from 'react';
/* External utilities */
import _find from 'lodash/find';

import {
    INITIAL,
    REFUNDED,
    REFUND_FAILED,
} from 'projects/depreacted/hotels/constants/hotelsBookingStatuses';
import {
    FISCAL_RECEIPT_ACQUIRE_TYPE,
    FISCAL_RECEIPT_CLEAR_TYPE,
    FISCAL_RECEIPT_REFUND_TYPE,
} from 'projects/depreacted/hotels/constants/fiscalReceiptTypes';
import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';

import {IOrderInfoReducer} from 'reducers/depreacted/hotels/bookAndPayPage/orderInfo/reducer';
import {IOfferInfoByTokenReducer} from 'reducers/depreacted/hotels/bookAndPayPage/offerInfoByToken/reducer';

import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';
import {CHAR_MIDDLE_DOT} from 'utilities/strings/charCodes';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getHotelsSearchUrlByParams} from 'projects/depreacted/hotels/utilities/urls/getHotelsSearchUrl';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';

import * as i18nBlock from 'i18n/hotels-CancelledOrderPage';

import Link from 'components/Link/Link';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import BookLoader from 'components/BookLoader/BookLoader';
import FiscalReceipt from 'projects/depreacted/hotels/components/FiscalReceipt/FiscalReceipt';
import BookHotelInfo from 'projects/depreacted/hotels/components/BookHotelInfo/BookHotelInfo';
import HotelsCancelledPenaltyInfo from 'projects/depreacted/hotels/components/HotelsCancelledPenaltyInfo/HotelsCancelledPenaltyInfo';
import HotelsBookLayout from 'projects/depreacted/hotels/components/HotelsBookLayout/HotelsBookLayout';

import BookRedirectController from 'projects/depreacted/hotels/containers/BookRedirectController/BookRedirectController';
import bookInfoProvider from 'projects/depreacted/hotels/containers/BookInfoProvider/BookInfoProvider';

import cx from './CancelledOrderPage.scss';

export interface ICancelledOrderPageProps
    extends IWithClassName,
        IWithDeviceType {
    history: {
        push: Function;
    };
    location: {
        pathname: string;
        search: string;
    };
    orderInfo: IOrderInfoReducer;
    offerInfoByToken: IOfferInfoByTokenReducer;
}

class CancelledOrderPage extends Component<ICancelledOrderPageProps> {
    static defaultProps = {
        offerInfoByToken: {
            isLoading: false,
        },
        deviceType: {},
        orderInfo: {},
    };

    /* Helpers */

    private preparePenaltyInfo(): ReactNode {
        const penalty = this.props.orderInfo.refundableInfo?.penalty;
        const penaltyIntervalIndex =
            this.props.orderInfo.refundableInfo?.penaltyIntervalIndex;

        const refundableRules =
            this.props.offerInfoByToken.offerInfo?.cancellationInfo
                ?.refundableRules;

        const refundableRule =
            penaltyIntervalIndex === undefined
                ? undefined
                : refundableRules?.[penaltyIntervalIndex];
        const startDate = refundableRule?.startDate;
        const type = refundableRule?.type;

        return (
            <HotelsCancelledPenaltyInfo
                className={cx('penaltyAmount')}
                type={type}
                penalty={penalty}
                startDate={startDate}
            />
        );
    }

    private checkRenderLoader = (): boolean => {
        const {
            location,
            orderInfo: {status},
            offerInfoByToken: {isLoading},
        } = this.props;

        const {orderId} = getQueryByLocation(location);

        return Boolean(isLoading || (orderId && status === INITIAL));
    };

    getSearchHotelsPagePath(): string | null {
        const geoRegions =
            this.props.offerInfoByToken.offerInfo?.hotelInfo.breadcrumbs
                ?.geoRegions;
        const searchParams =
            this.props.offerInfoByToken.offerInfo?.searchParams;

        if (!geoRegions || !searchParams) {
            return null;
        }

        return getHotelsSearchUrlByParams(
            geoRegions[geoRegions.length - 1].geoId,
            {
                ...searchParams,
                checkinDate: searchParams.startDate,
                checkoutDate: searchParams.endDate,
            },
        );
    }

    /* Page Loader */

    private renderPageLoader(): React.ReactNode {
        const canRenderPageLoader = this.checkRenderLoader();

        return (
            <BookLoader
                title={i18nBlock.orderLoadingDotTitle()}
                description={i18nBlock.orderLoadingDotDescription()}
                isLoading={canRenderPageLoader}
            />
        );
    }

    /* HotelCard Block */

    private renderHotelInfo(): React.ReactNode {
        const {
            offerInfoByToken: {offerInfo},
            orderInfo: {status},
            deviceType,
        } = this.props;

        const roomInfo = offerInfo?.roomInfo;
        const hotelInfo = offerInfo?.hotelInfo;
        const bedsGroups = offerInfo?.bedsGroups ?? [];
        const searchParams = offerInfo?.searchParams;
        const cancellationInfo = offerInfo?.cancellationInfo;
        const partnerHotelInfo = offerInfo?.partnerHotelInfo;
        const mealInfo = offerInfo?.mealInfo;

        const firstBedsGroup = bedsGroups.slice(0, 1);
        const footerNode = status === REFUNDED && this.renderSearchButtons();

        return (
            <BookHotelInfo
                className={cx('hotelInfo')}
                deviceType={deviceType}
                roomInfo={roomInfo}
                hotelInfo={hotelInfo}
                mealInfo={mealInfo}
                searchParams={searchParams}
                bedsGroups={firstBedsGroup}
                partnerHotelInfo={partnerHotelInfo}
                cancellationInfo={cancellationInfo}
                preventOpacityOverlay={status === REFUND_FAILED}
                footerNode={footerNode}
                isExpiredOffer
            />
        );
    }

    /* Main Content Block */

    private renderRefundedOrder(): React.ReactNode {
        const {orderInfo} = this.props;

        const customerEmail = orderInfo.guestsInfo?.customerEmail;

        const yandexOrderId = orderInfo.confirmationInfo?.yandexOrderId ?? '';
        const confirmationId = orderInfo.confirmationInfo?.confirmationId ?? '';

        const receipts = orderInfo.payment?.receipts;

        const penalty = orderInfo.refundableInfo?.penalty;

        const acquireFiscalReceipt = _find(receipts, {
            type: FISCAL_RECEIPT_ACQUIRE_TYPE,
        });
        const clearOrRefundFiscalReceipt = _find(
            receipts,
            ({type}) =>
                type === FISCAL_RECEIPT_CLEAR_TYPE ||
                type === FISCAL_RECEIPT_REFUND_TYPE,
        );

        const penaltyInfoTitle = this.preparePenaltyInfo();

        return (
            <section className={cx('cancellationInfo')}>
                <div className={cx('mainCancellationInfo')}>
                    <h2 className={cx('refundableTitle')}>
                        {!penalty || parseFloat(String(penalty.amount)) === 0
                            ? i18nBlock.freeRefundableTitle()
                            : i18nBlock.refundableTitle()}
                    </h2>
                    <div className={cx('confirmationMail')}>
                        {insertJSXIntoKey(i18nBlock.confirmationMail)({
                            userEmail: (
                                <span
                                    className={cx('customerEmail')}
                                    key={customerEmail}
                                >
                                    {customerEmail}
                                </span>
                            ),
                        })}
                    </div>
                    {penaltyInfoTitle && (
                        <div className={cx('penaltyInfo')}>
                            {penaltyInfoTitle}
                        </div>
                    )}
                    <div className={cx('receipts')}>
                        <FiscalReceipt
                            canRenderTooltipForEmptyUrl={true}
                            triggerText={i18nBlock.frtAquire()}
                            fiscalReceipt={acquireFiscalReceipt}
                        />
                        <span className={cx('dotSplitter')}>
                            {CHAR_MIDDLE_DOT}
                        </span>
                        <FiscalReceipt
                            canRenderTooltipForEmptyUrl={true}
                            triggerText={i18nBlock.frtRefund()}
                            fiscalReceipt={clearOrRefundFiscalReceipt}
                        />
                    </div>
                    <div className={cx('confirmationOrderIds')}>
                        {i18nBlock.orderIds({
                            yandexOrderId,
                            confirmationId,
                        })}
                    </div>
                </div>
                {this.renderHotelInfo()}
            </section>
        );
    }

    /* SearchButtons for HotelCard Footer */

    private renderSearchButtons(): React.ReactNode {
        const searchHotelsPagePath = this.getSearchHotelsPagePath();

        return (
            <div className={cx('searchButtons')}>
                <ButtonLink url={URLs.account}>
                    <span className={cx('searchButtonText')}>
                        {i18nBlock.searchOrdersButton()}
                    </span>
                </ButtonLink>
                {searchHotelsPagePath && (
                    <ButtonLink
                        className={cx('searchButton')}
                        theme="primary"
                        url={searchHotelsPagePath}
                    >
                        <span className={cx('searchButtonText')}>
                            {i18nBlock.searchHotelsButton()}
                        </span>
                    </ButtonLink>
                )}
            </div>
        );
    }

    /* Failed Order Block */

    renderRefundFailedOrder(): React.ReactNode {
        const yandexOrderId =
            this.props.orderInfo.confirmationInfo?.yandexOrderId;

        // Вернуть, когда будет работать поддержка по телефону
        // const supportPhoneNode = (
        //     <SupportPhone
        //         className={cx('errorSupportPhone')}
        //     />
        // );

        const supportLinkNode = (
            <Link
                url={i18nBlock.supportLinkHref()}
                target="_blank"
                rel="noopener noreferrer"
                tabIndex={-1}
                text={i18nBlock.supportLinkTitle()}
            />
        );

        const orderIdNode = (
            <span
                className={cx('errorOrderId')}
                {...prepareQaAttributes({
                    current: 'errorOrderId',
                })}
            >
                {yandexOrderId}
            </span>
        );

        return (
            <div className={cx('errorContainer')}>
                <section className={cx('errorSection')}>
                    <h4
                        className={cx('errorSectionTitle')}
                        {...prepareQaAttributes({
                            current: 'errorSectionTitle',
                        })}
                    >
                        {i18nBlock.refundFailedDotTitle()}
                    </h4>
                    <div className={cx('errorSectionDescription')}>
                        {i18nBlock.refundFailedDotDescription()}
                    </div>
                    <div className={cx('errorSectionItem')}>
                        {insertJSXIntoKey(i18nBlock.supportFullText)({
                            supportLinkNode,
                        })}
                    </div>
                    <div className={cx('errorSectionItem')}>
                        {insertJSXIntoKey(i18nBlock.supportOrderId)({
                            orderId: orderIdNode,
                        })}
                    </div>
                </section>
                {this.renderHotelInfo()}
            </div>
        );
    }

    /* Page Content Block */

    renderPageContent(): React.ReactNode {
        const {
            orderInfo: {status},
        } = this.props;

        switch (status) {
            case REFUNDED: {
                return this.renderRefundedOrder();
            }

            case REFUND_FAILED: {
                return this.renderRefundFailedOrder();
            }

            default: {
                return null;
            }
        }
    }

    render(): React.ReactNode {
        const {deviceType} = this.props;
        const canRenderPageLoader = this.checkRenderLoader();

        return (
            <HotelsBookLayout
                deviceType={deviceType}
                hasLoader={canRenderPageLoader}
            >
                <BookRedirectController />
                <div
                    className={cx(
                        'cancelledOrderPage',
                        deviceMods('cancelledOrderPage', deviceType),
                    )}
                    {...prepareQaAttributes({
                        current: 'hotelsCancelledOrderPage',
                    })}
                >
                    {this.renderPageLoader()}
                    {this.renderPageContent()}
                </div>
            </HotelsBookLayout>
        );
    }
}

export default bookInfoProvider({
    canFetchOfferInfo: false,
    canStartOrderPolling: true,
})(CancelledOrderPage);
