import {PureComponent, ReactNode} from 'react';

import {ISearchFormBlock} from 'types/hotels/geoRegion/IGeoRegionInfo';
import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {ESearchFormFieldName} from 'components/SearchForm/types';

import {deviceMods} from 'utilities/stylesUtils';

import Heading from 'components/Heading/Heading';
import SearchForm from 'projects/depreacted/hotels/components/SearchForm/SearchForm';

import cx from './SearchFormBlock.scss';

interface ISearchFormBlockProps extends IWithClassName, IWithDeviceType {
    block: ISearchFormBlock;
}

class SearchFormBlock extends PureComponent<ISearchFormBlockProps> {
    renderSearchForm(): ReactNode {
        const {
            block: {
                data: {
                    searchFormParams: {isCalendarOpen},
                    filterParams,
                },
            },
        } = this.props;

        const filterAtoms = filterParams?.filterAtoms;
        const searchUrlAdditionalQuery = filterAtoms
            ? {filterAtoms: filterAtoms.join(',')}
            : undefined;

        return (
            <SearchForm
                className={cx('searchForm')}
                autoFocusFieldName={
                    isCalendarOpen ? ESearchFormFieldName.START_DATE : undefined
                }
                queryParams={searchUrlAdditionalQuery}
            />
        );
    }

    renderDesktopSearchFormContainer(): ReactNode {
        const {
            block: {
                data: {title, imageUrl},
            },
        } = this.props;

        return (
            <>
                <img
                    className={cx('geoRegionMainImage')}
                    src={imageUrl}
                    alt={title}
                />
                <div className={cx('searchFormCard')}>
                    {this.renderSearchFormWithTitle()}
                </div>
            </>
        );
    }

    renderMobileSearchFormContainer(): ReactNode {
        return this.renderSearchFormWithTitle();
    }

    renderSearchFormWithTitle(): ReactNode {
        const {
            block: {
                data: {title},
            },
            deviceType,
        } = this.props;

        return (
            <>
                <Heading
                    level="1"
                    className={cx(
                        'mainTitle',
                        deviceMods('mainTitle', deviceType),
                    )}
                >
                    {title}
                </Heading>
                {this.renderSearchForm()}
            </>
        );
    }

    render(): ReactNode {
        const {
            className,
            deviceType: {isMobile},
        } = this.props;

        return (
            <div className={cx(className, 'searchFormBlock')}>
                {isMobile
                    ? this.renderMobileSearchFormContainer()
                    : this.renderDesktopSearchFormContainer()}
            </div>
        );
    }
}

export default SearchFormBlock;
