import React, {useCallback, useEffect, useMemo} from 'react';
import {useHistory, useLocation} from 'react-router-dom';
import {createPath} from 'history';

import {EProjectName} from 'constants/common';
import {
    MC_HOTELS_VOUCHER_DOWNLOAD_PARAMS,
    MC_HOTELS_VOUCHER_PRINT_PARAMS,
} from 'constants/metrika/hotels';
import {PLUS_BONUS_POINTS_VESTING_PERIOD} from 'constants/plus';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EFooterProject} from 'components/Footer/types';
import {EYandexPlusApplicationMode} from 'types/hotels/offer/IHotelOffer';

import {getOrderUrl} from 'projects/account/utilities/urls/getOrderUrl';
import {getHotelsVoucherUrl} from 'projects/depreacted/hotels/utilities/getHotelsVoucherUrl/getHotelsVoucherUrl';
import {getPathByOrderStatus} from 'projects/depreacted/hotels/utilities/urls/getPathByOrderStatus';
import {reachGoal} from 'utilities/metrika';
import {useReachGoal} from 'utilities/metrika/useReachGoal';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {prepareDeferredPaymentDateTime} from 'projects/depreacted/hotels/utilities/prepareDeferredPaymentDate/prepareDeferredPaymentDate';
import {HUMAN_DATETIME_WITHOUT_YEAR} from 'utilities/dateUtils/formats';
import {useHotelVoucherStatus} from 'utilities/hooks/useHotelVoucherStatus';

import * as i18nBlock from 'i18n/happyPage';
import * as i18nHotelsPlusBlock from 'i18n/hotels-plusPromo2021';

import HappyPageLayout from 'components/Layouts/HappyPageLayout/HappyPageLayout';
import OrderInfo from 'projects/depreacted/hotels/pages/HappyPage/components/OrderInfo/OrderInfo';
import SingleOrderLayout from 'components/Layouts/HappyPageLayout/components/SingleOrderLayout/SingleOrderLayout';
import OrderActions from 'components/OrderActions/OrderActions';
import OrderHeader from 'components/OrderHeader/OrderHeader';
import Separator from 'components/Separator/Separator';
import BudapeshtBanner from 'projects/depreacted/hotels/pages/HappyPage/components/BudapeshtBanner/BudapeshtBanner';
import PlusBanner from 'projects/happyPage/components/PlusBanner/PlusBanner';
import NextPaymentInfo from 'projects/depreacted/hotels/pages/HappyPage/components/NextPaymentInfo/NextPaymentInfo';
import Spinner from 'components/Spinner/Spinner';

import cx from './HappyPageContent.scss';

const HAPPY_PAGE_QA = 'happyPage';

interface IHappyPageContentProps {
    orderId: string;
    orderAndCrossSalesInfo: any;
}

const HappyPageContent: React.FC<IHappyPageContentProps> = props => {
    const {orderId, orderAndCrossSalesInfo} = props;

    const location = useLocation();
    const history = useHistory();
    const deviceType = useDeviceType();

    const plusInfo =
        orderAndCrossSalesInfo.order.appliedPromoCampaigns?.yandexPlus;
    const isYandexHotel =
        orderAndCrossSalesInfo.order.offerInfo.hotelInfo.isYandexHotel;

    const {
        isReady: isVoucherReady,
        isLoading: isVoucherStatusLoading,
        isError: isVoucherStatusError,
    } = useHotelVoucherStatus(orderId);

    const reachOrderShownGoal = useReachGoal(
        EHotelsGoal.HOTEL_HAPPY_PAGE_ORDER_SHOWN,
    );
    const reachDetailsClick = useReachGoal(
        EHotelsGoal.HOTEL_HAPPY_PAGE_ORDER_DETAILS_CLICK,
    );
    const reachPrintClick = useReachGoal(
        EHotelsGoal.HAPPY_PAGE_VOUCHER_BUTTON,
        [MC_HOTELS_VOUCHER_PRINT_PARAMS],
    );
    const reachDownloadClick = useReachGoal(
        EHotelsGoal.HAPPY_PAGE_VOUCHER_BUTTON,
        [MC_HOTELS_VOUCHER_DOWNLOAD_PARAMS],
    );

    const headerTitle = useMemo(() => {
        const hasNextPayment = Boolean(
            orderAndCrossSalesInfo.order.payment.next,
        );
        const usesDeferredPayments =
            orderAndCrossSalesInfo.order.payment.usesDeferredPayments;

        if (usesDeferredPayments && !hasNextPayment) {
            return i18nBlock.hotelSuccessPaymentTitle();
        }

        return i18nBlock.hotelTitle();
    }, [orderAndCrossSalesInfo]);

    const header = useMemo(() => {
        const {
            order: {
                confirmationInfo: {yandexOrderId},
            },
        } = orderAndCrossSalesInfo;

        return (
            <OrderHeader
                title={headerTitle}
                type={EProjectName.HOTELS}
                number={yandexOrderId}
                showSuccessBadge
                {...prepareQaAttributes('orderHeader')}
            />
        );
    }, [headerTitle, orderAndCrossSalesInfo]);

    const banner = useMemo(() => {
        const isPlusTopup = Boolean(
            plusInfo?.mode === EYandexPlusApplicationMode.TOPUP &&
                plusInfo?.points,
        );

        return (
            <>
                {isYandexHotel && <BudapeshtBanner deviceType={deviceType} />}
                {isPlusTopup && (
                    <PlusBanner
                        className={cx(
                            'plusBanner',
                            deviceMods('plusBanner', deviceType),
                        )}
                        deviceType={deviceType}
                        text={i18nHotelsPlusBlock.happyPageBannerText({
                            days: PLUS_BONUS_POINTS_VESTING_PERIOD,
                            plusPoints: plusInfo?.points || 0,
                        })}
                    />
                )}
            </>
        );
    }, [deviceType, isYandexHotel, plusInfo]);

    const nextPaymentInfo = useMemo(() => {
        const nextPayment = orderAndCrossSalesInfo.order.payment.next;

        if (!nextPayment) {
            return null;
        }

        return (
            <>
                <NextPaymentInfo
                    url={getOrderUrl(orderId)}
                    className={cx(deviceMods('nextPaymentInfo', deviceType))}
                    deviceType={deviceType}
                    price={nextPayment.amount}
                    dateText={prepareDeferredPaymentDateTime(
                        nextPayment.paymentEndsAt,
                        HUMAN_DATETIME_WITHOUT_YEAR,
                    )}
                    {...prepareQaAttributes(HAPPY_PAGE_QA)}
                />
                {deviceType.isMobile && <Separator margin={4} />}
            </>
        );
    }, [deviceType, orderAndCrossSalesInfo, orderId]);

    const orderInfo = useMemo(() => {
        const {
            order: {
                guestsInfo: {customerEmail, customerPhone},
            },
        } = orderAndCrossSalesInfo;

        return (
            <SingleOrderLayout
                orderInfo={
                    <OrderInfo
                        order={orderAndCrossSalesInfo.order}
                        onMount={reachOrderShownGoal}
                        {...prepareQaAttributes(HAPPY_PAGE_QA)}
                    />
                }
                description={
                    isYandexHotel ? '' : i18nBlock.hotelInfoSentToEmail()
                }
                email={customerEmail}
                phone={customerPhone}
                {...prepareQaAttributes(HAPPY_PAGE_QA)}
            />
        );
    }, [isYandexHotel, orderAndCrossSalesInfo, reachOrderShownGoal]);

    const renderSpinnerIcon = useCallback(
        (className?: string): JSX.Element => (
            <Spinner className={className} size="xxs" />
        ),
        [],
    );

    const orderActions = useMemo(() => {
        const {
            order: {
                confirmationInfo: {yandexOrderId},
            },
        } = orderAndCrossSalesInfo;

        return (
            <OrderActions
                orderDetails={{
                    url: getOrderUrl(orderId),
                    onClick: reachDetailsClick,
                }}
                print={{
                    title: i18nBlock.print(),
                    url: getHotelsVoucherUrl(yandexOrderId, {
                        actionType: 'DownloadActionTypes.PRINT',
                        searchBy: 'ESearchByOrderId.PRETTY_ORDER_ID',
                    }),
                    renderIcon: isVoucherStatusLoading
                        ? renderSpinnerIcon
                        : undefined,
                    isDisabled: !isVoucherReady,
                    onClick: reachPrintClick,
                }}
                download={{
                    title: i18nBlock.download(),
                    url: getHotelsVoucherUrl(yandexOrderId, {
                        actionType: 'DownloadActionTypes.DOWNLOAD',
                        searchBy: 'ESearchByOrderId.PRETTY_ORDER_ID',
                    }),
                    renderIcon: isVoucherStatusLoading
                        ? renderSpinnerIcon
                        : undefined,
                    isDisabled: !isVoucherReady,
                    onClick: reachDownloadClick,
                }}
                {...prepareQaAttributes('orderActions')}
            />
        );
    }, [
        orderAndCrossSalesInfo,
        orderId,
        reachDetailsClick,
        isVoucherStatusLoading,
        renderSpinnerIcon,
        isVoucherReady,
        reachPrintClick,
        reachDownloadClick,
    ]);

    useEffect(() => {
        const {
            order: {status, orderCancellationDetails},
        } = orderAndCrossSalesInfo;

        const currentPath = createPath(location);
        const path = getPathByOrderStatus({
            orderId,
            orderStatus: status,
            orderCancellationDetails,
            location,
        });

        if (path && currentPath !== path) {
            history.replace(path);
        }
    }, [orderId, orderAndCrossSalesInfo, location, history]);

    useEffect(() => {
        if (isVoucherReady) {
            reachGoal(EHotelsGoal.HAPPY_PAGE_VOUCHER_SUCCESS);
        }
    }, [isVoucherReady]);

    useEffect(() => {
        if (isVoucherStatusError) {
            reachGoal(EHotelsGoal.HAPPY_PAGE_VOUCHER_ERROR);
        }
    }, [isVoucherStatusError]);

    return (
        <HappyPageLayout
            project={EProjectName.HOTELS}
            footerType={EFooterProject.HOTELS_BOOKING}
            header={header}
            banner={banner}
            orderHeaderInfo={nextPaymentInfo}
            orderInfo={orderInfo}
            orderActions={orderActions}
            crossSalesClassName={cx(deviceMods('crossSales', deviceType))}
        />
    );
};

export default React.memo(HappyPageContent);
