import React from 'react';

import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'types/withClassName';

import IPrice from 'utilities/currency/PriceInterface';
import {deviceModDesktop, deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-NextPaymentInfo';

import Flex from 'components/Flex/Flex';
import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';

import cx from './NextPaymentInfo.scss';

interface INextPaymentInfoProps
    extends IWithDeviceType,
        IWithClassName,
        IWithQaAttributes {
    price: IPrice;
    dateText: string;
    url: string;
}

const NextPaymentInfo: React.FC<INextPaymentInfoProps> = ({
    url,
    deviceType,
    className,
    price,
    dateText,
    ...restProps
}) => {
    return (
        <Flex
            className={cx('root', className)}
            justifyContent="space-between"
            alignItems="center"
            flexDirection={deviceType.isMobile ? 'column' : 'row'}
        >
            <Flex className={cx('info')} flexDirection="column">
                <Flex justifyContent="space-between">
                    <Text className={cx('label')}>
                        {i18nBlock.remainingPrice()}
                    </Text>
                    <Price
                        className={cx(
                            'price',
                            deviceModDesktop('price', deviceType),
                        )}
                        value={price.value}
                        currency={price.currency}
                        {...prepareQaAttributes({
                            parent: restProps,
                            current: 'nextPaymentPrice',
                        })}
                    />
                </Flex>
                <Flex className={cx('dateInfo')} justifyContent="space-between">
                    <Text className={cx('label')}>
                        {i18nBlock.paymentPeriod()}
                    </Text>
                    <Text
                        align="right"
                        size={deviceType.isDesktop ? 'l' : 'm'}
                        weight="bold"
                        {...prepareQaAttributes({
                            parent: restProps,
                            current: 'paymentEndsAt',
                        })}
                    >
                        {i18nBlock.payTillText({date: dateText})}
                    </Text>
                </Flex>
            </Flex>
            <ButtonLink
                url={url}
                target="_blank"
                rel="noopener noreferrer"
                size={deviceType.isMobile ? 'l' : undefined}
                className={cx(deviceMods('button', deviceType))}
                {...prepareQaAttributes({
                    parent: restProps,
                    current: 'nextPaymentLink',
                })}
            >
                {i18nBlock.payButtonText()}
            </ButtonLink>
        </Flex>
    );
};

export default NextPaymentInfo;
