import React, {PureComponent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {
    parseImageUrlTemplate,
    convertUrlTemplateImagesToFlatList,
} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';

import GalleryRow from './GalleryRow/GalleryRow';

import cx from './Gallery.scss';

/* Constants */
const MIN_IMAGES_COUNT = 1;
const MAX_IMAGES_COUNT = 4;

interface IHotelsGalleryProps extends IWithClassName {
    images: IHotelOrRoomImage[];
    onMoreImagesClick: (index: number) => void;
    totalImageCount: number;
    imagesAlt?: string;
}

/**
 * Галерея главного таба страницы отеля.
 */
class HotelsGallery extends PureComponent<IHotelsGalleryProps> {
    static defaultProps = {
        images: [],
        className: '',
        children: null,
        totalImageCount: 0,
    };

    private handleClickOnBigRow = () => this.props.onMoreImagesClick(0);

    private handleClickSmallRow = (index: number) =>
        this.props.onMoreImagesClick(index + 1);

    private renderNoImagesLayout(): React.ReactNode {
        return null;
    }

    private renderBigRow(image: IHotelOrRoomImage): React.ReactNode {
        const {imagesAlt} = this.props;

        const sizedImageUrl = parseImageUrlTemplate(
            image.urlTemplate,
            EHotelImageAltayStandardSize.XL,
            image.sizes,
        );

        return (
            <img
                key={sizedImageUrl}
                alt={imagesAlt}
                src={sizedImageUrl}
                className={cx('bigRow')}
                onClick={this.handleClickOnBigRow}
                loading="lazy"
            />
        );
    }

    private renderSmallRow(images: IHotelOrRoomImage[]): React.ReactNode {
        const {totalImageCount, imagesAlt} = this.props;

        const sizedImagesUrls = convertUrlTemplateImagesToFlatList(
            images,
            EHotelImageAltayStandardSize.M,
        );

        return (
            <GalleryRow
                isLastClickable={totalImageCount > MAX_IMAGES_COUNT}
                imageSources={sizedImagesUrls}
                className={cx('smallRow')}
                clickableWrapperClassName={cx('clickable')}
                totalImageCount={totalImageCount}
                onImageClick={this.handleClickSmallRow}
                onMoreImagesClick={this.handleClickSmallRow}
                imagesAlt={imagesAlt}
            />
        );
    }

    private renderTileLayout = (): React.ReactNode => {
        const {images, className} = this.props;

        const canRenderBigRow =
            images.length === MIN_IMAGES_COUNT ||
            images.length >= MAX_IMAGES_COUNT;
        const canRenderSmallRow = images.length > MIN_IMAGES_COUNT;

        const qa = prepareQaAttributes('hotelPageGallery');

        if (images.length > 0) {
            if (canRenderBigRow) {
                const [bigImage, ...smallImages] = images;

                return (
                    <Box between={1} className={className} tagProps={qa}>
                        {this.renderBigRow(bigImage)}
                        {canRenderSmallRow && this.renderSmallRow(smallImages)}
                    </Box>
                );
            } else if (canRenderSmallRow) {
                return (
                    <Box between={1} className={className} tagProps={qa}>
                        {this.renderSmallRow(images)}
                    </Box>
                );
            }
        }

        return this.renderNoImagesLayout();
    };

    render(): React.ReactNode {
        return <div className={cx('root')}>{this.renderTileLayout()}</div>;
    }
}

export default HotelsGallery;
