import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import Box from 'components/Box/Box';
import Image from 'components/TravelImage/TravelImage';

import cx from './GalleryRow.scss';

interface IGalleryRowProps extends IWithClassName {
    imageContainerClassName?: string;
    imageSources: string[];
    clickableWrapperClassName?: string;
    countClassName?: string;
    isLastClickable?: boolean;
    totalImageCount?: number;
    onMoreImagesClick?: (imageIndex: number) => void;
    onImageClick: (imageIndex: number) => void;
    customRenderOverlay?: () => React.ReactNode;
    imagesAlt?: string;
}

const GalleryRow: React.FC<IGalleryRowProps> = props => {
    const {
        imageSources,
        totalImageCount,
        onMoreImagesClick,
        clickableWrapperClassName,
        countClassName,
        isLastClickable,
        className,
        imageContainerClassName,
        onImageClick,
        customRenderOverlay,
        imagesAlt,
    } = props;

    const handleMoreImagesClick = useCallback(() => {
        if (onMoreImagesClick) {
            onMoreImagesClick(imageSources.length - 1);
        }
    }, [onMoreImagesClick, imageSources]);

    const handleImageClick = useCallback(
        (e, index?: number | string) => {
            if (onImageClick && typeof index === 'number') {
                onImageClick(index);
            }
        },
        [onImageClick],
    );

    function renderOverlay(): React.ReactNode {
        return (
            <div className={cx('overlay')}>
                <div className={cx('text')}>
                    <p className={cx('line', 'line-count', countClassName)}>
                        {totalImageCount || ''}
                    </p>
                    <p className={cx('line')}>{i18nBlock.viewAllImages()}</p>
                </div>
            </div>
        );
    }

    function renderImage(src: string, index: number): React.ReactNode {
        const image = (
            <Image
                key={index + src}
                id={index}
                hasFitCover
                className={cx(imageContainerClassName, 'container')}
                imageClassName={cx('image')}
                src={src}
                onClick={handleImageClick}
                imageAlt={imagesAlt}
            />
        );

        if (isLastClickable && index === imageSources.length - 1) {
            return (
                <div
                    key="last-clickable"
                    className={cx(
                        'image-wrapper_clickable',
                        clickableWrapperClassName,
                    )}
                    onClick={handleMoreImagesClick}
                >
                    {image}
                    {customRenderOverlay
                        ? customRenderOverlay()
                        : renderOverlay()}
                </div>
            );
        }

        return image;
    }

    return (
        <Box className={className} nowrap inline between={1}>
            {imageSources.map(renderImage)}
        </Box>
    );
};

export default GalleryRow;
