import React, {PureComponent} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {
    parseImageUrlTemplate,
    convertUrlTemplateImagesToFlatList,
} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';

import Box from 'components/Box/Box';

import GalleryTable from './GalleryTable/GalleryTable';

import cx from './GalleryRedesign.scss';

/* Constants */
const MIN_IMAGES_COUNT = 1;
const MAX_IMAGES_COUNT = 5;

interface IHotelsGalleryProps extends IWithClassName {
    images: IHotelOrRoomImage[];
    onMoreImagesClick: (index: number) => void;
    totalImageCount: number;
    imagesAlt?: string;
}

/**
 * Галерея главного таба страницы отеля.
 */
class HotelsGalleryRedesign extends PureComponent<IHotelsGalleryProps> {
    static defaultProps = {
        images: [],
        className: '',
        children: null,
        totalImageCount: 0,
    };

    private handleClickOnBigPicture = () => this.props.onMoreImagesClick(0);

    private handleClickTableElement = (index: number) =>
        this.props.onMoreImagesClick(index + 1);

    private renderBigPicture(
        image: IHotelOrRoomImage,
        columnClassName: string,
    ): React.ReactNode {
        const {imagesAlt} = this.props;

        const sizedImageUrl = parseImageUrlTemplate(
            image.urlTemplate,
            EHotelImageAltayStandardSize.XL,
            image.sizes,
        );

        return (
            <img
                key={sizedImageUrl}
                alt={imagesAlt}
                src={sizedImageUrl}
                className={cx('bigPicture', columnClassName)}
                onClick={this.handleClickOnBigPicture}
            />
        );
    }

    private renderSmallTable(
        images: IHotelOrRoomImage[],
        columnClassName: string,
    ): React.ReactNode {
        const {totalImageCount, imagesAlt} = this.props;

        const sizedImagesUrls = convertUrlTemplateImagesToFlatList(
            images,
            EHotelImageAltayStandardSize.M,
        );

        return (
            <GalleryTable
                isLastClickable={totalImageCount > MAX_IMAGES_COUNT}
                imageSources={sizedImagesUrls}
                className={cx(columnClassName)}
                totalImageCount={totalImageCount}
                onImageClick={this.handleClickTableElement}
                onMoreImagesClick={this.handleClickTableElement}
                imagesAlt={imagesAlt}
            />
        );
    }

    private renderTileLayout = (): React.ReactNode => {
        const {images} = this.props;
        const canRenderRightSubTable = images.length > MIN_IMAGES_COUNT;

        const qa = prepareQaAttributes('hotelPageGallery');

        if (images.length > 0) {
            const [bigImage, ...smallImages] = images;
            const columnClassName = canRenderRightSubTable
                ? 'twoColumn'
                : 'oneColumn';

            return (
                <Box
                    className={cx('galleryWrapper')}
                    between={1}
                    inline
                    nowrap
                    tagProps={qa}
                >
                    {this.renderBigPicture(bigImage, columnClassName)}
                    {canRenderRightSubTable &&
                        this.renderSmallTable(smallImages, columnClassName)}
                </Box>
            );
        }

        return null;
    };

    render(): React.ReactNode {
        const {className} = this.props;

        return (
            <div className={cx('root', className)}>
                {this.renderTileLayout()}
            </div>
        );
    }
}

export default HotelsGalleryRedesign;
