import React, {useCallback} from 'react';

import {IWithClassName} from 'types/withClassName';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import Image from 'components/TravelImage/TravelImage';
import Box from 'components/Box/Box';

import cx from './GalleryTable.scss';

interface IGalleryRowProps extends IWithClassName {
    imageSources: string[];
    isLastClickable?: boolean;
    totalImageCount: number;
    onMoreImagesClick?: (imageIndex: number) => void;
    onImageClick: (imageIndex: number) => void;
    customRenderOverlay?: () => React.ReactNode;
    imagesAlt?: string;
}

const GalleryTable: React.FC<IGalleryRowProps> = props => {
    const {
        imageSources,
        totalImageCount,
        onMoreImagesClick,
        isLastClickable,
        className,
        onImageClick,
        customRenderOverlay,
        imagesAlt,
    } = props;

    const handleMoreImagesClick = useCallback(() => {
        if (onMoreImagesClick) {
            onMoreImagesClick(imageSources.length - 1);
        }
    }, [onMoreImagesClick, imageSources]);

    const handleImageClick = useCallback(
        (e, index?: number | string) => {
            if (onImageClick && typeof index === 'number') {
                onImageClick(index);
            }
        },
        [onImageClick],
    );

    function renderOverlay(): React.ReactNode {
        return (
            <div className={cx('overlay')}>
                <div className={cx('text')}>
                    <p className={cx('line', 'line-count')}>
                        {totalImageCount || ''}
                    </p>
                    <p className={cx('line')}>{i18nBlock.viewAllImages()}</p>
                </div>
            </div>
        );
    }

    function renderImage(index: number, halfHeight = false): React.ReactNode {
        const src = imageSources[index];
        const needClickableWrapper =
            isLastClickable && index === imageSources.length - 1;

        const image = (
            <Image
                key={index + src}
                id={index}
                hasFitCover
                className={cx(
                    'container',
                    halfHeight &&
                        !needClickableWrapper &&
                        'container_halfHeight',
                )}
                imageClassName={cx('image')}
                src={src}
                onClick={handleImageClick}
                imageAlt={imagesAlt}
                isRenderImmediately
            />
        );

        if (needClickableWrapper) {
            return (
                <div
                    key="last-clickable"
                    className={cx('image-wrapper_clickable')}
                    onClick={handleMoreImagesClick}
                >
                    {image}
                    {customRenderOverlay
                        ? customRenderOverlay()
                        : renderOverlay()}
                </div>
            );
        }

        return image;
    }

    let content;

    switch (imageSources.length) {
        case 1:
            content = (
                <Box className={cx('column', 'column_one')} nowrap inline>
                    {renderImage(0)}
                </Box>
            );

            break;
        case 2:
            content = (
                <>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(0)}
                    </Box>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(1)}
                    </Box>
                </>
            );

            break;
        case 3:
            content = (
                <>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(0)}
                    </Box>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(1, true)}
                        {renderImage(2, true)}
                    </Box>
                </>
            );

            break;
        default:
            content = (
                <>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(0, true)}
                        {renderImage(1, true)}
                    </Box>
                    <Box className={cx('column', 'column_two')} nowrap inline>
                        {renderImage(2, true)}
                        {renderImage(3, true)}
                    </Box>
                </>
            );

            break;
    }

    return <div className={cx('root', className)}>{content}</div>;
};

export default GalleryTable;
