import React, {
    FunctionComponent,
    useMemo,
    useCallback,
    useState,
    useEffect,
    useRef,
} from 'react';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';
import {IHotel} from 'types/hotels/hotel/IHotel';

import {reachGoal} from 'utilities/metrika';
import {
    IGroupedImagesWithWidth,
    IImageUrlAndWidthResultType,
    convertAndGroupImagesForRows,
    MIN_PROPORTIONS_SUM,
} from './utilities/convertAndGroupImagesForRows';
import scrollTo from 'utilities/dom/scrollTo';

/* Components */
import LazyLoad from 'projects/depreacted/hotels/components/LazyLoad/LazyLoad';
import Spinner from 'components/Spinner/Spinner';
import Modal from 'components/Modal/Modal';
import HotelPageImagesViewer from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageImagesViewer/HotelPageImagesViewer';

import HotelImagesImage from '../HotelImagesImage/HotelImagesImage';
import HotelImagesTitle from '../HotelImagesTitle/HotelImagesTitle';

/* Init styles */

import cx from './HotelImagesDesktop.scss';

const HOTEL_IMAGE_SIZE_NAME = EHotelImageAltayStandardSize.L;
const ITEMS_TO_LOAD = 16;
const MAX_ITEMS_LEFT = 20;

/* Component Types */
interface IHotelImagesDesktopProps {
    isLoading: boolean;
    isVisible: boolean;
    images: IHotelOrRoomImage[];
    totalImageCount: number;
    hotel: IHotel;
    onMoreImages: (count: number) => void;
    onCloseClick: () => void;
}

const HotelImagesDesktop: FunctionComponent<IHotelImagesDesktopProps> =
    props => {
        const {
            images,
            isLoading,
            totalImageCount,
            onMoreImages,
            hotel,
            isVisible,
            onCloseClick,
        } = props;
        const {name: hotelName} = hotel;

        const [isViewerVisible, setViewerVisible] = useState(false);
        const [viewerImageIndex, setViewerIndex] = useState<number>();

        const scrollableRef = useRef<HTMLDivElement>(null);

        useEffect(() => {
            if (scrollableRef.current) {
                scrollTo({top: 0}, scrollableRef.current);
            }
        }, [isVisible]);

        const loadImageCount = useMemo(() => {
            if (totalImageCount - images.length < MAX_ITEMS_LEFT) {
                return totalImageCount - images.length;
            }

            return ITEMS_TO_LOAD;
        }, [images, totalImageCount]);

        const groupedImages = useMemo(
            () => convertAndGroupImagesForRows(images, HOTEL_IMAGE_SIZE_NAME),
            [images],
        );

        const handleImageClick = useCallback(
            (index: number) => {
                setViewerIndex(index);
                setViewerVisible(true);

                reachGoal(EHotelsGoal.HOTEL_PAGE_IMAGES_TAB_IMAGE_CLICK);
            },
            [setViewerIndex, setViewerVisible],
        );

        const handleCloseClick = useCallback(() => {
            setViewerVisible(false);
        }, [setViewerVisible]);

        const handleMoreImages = useCallback(() => {
            onMoreImages(loadImageCount);
        }, [onMoreImages, loadImageCount]);

        const renderImage = (
            image: IImageUrlAndWidthResultType,
        ): React.ReactNode => {
            return (
                <HotelImagesImage
                    className={cx('item')}
                    imageClassName={cx('image')}
                    style={{width: `${image.width}%`}}
                    src={image.src}
                    imageAlt={hotelName}
                    key={image.src + image.index}
                    imageIndex={image.index}
                    onImageClick={handleImageClick}
                />
            );
        };

        const renderRow = (
            imageRow: IGroupedImagesWithWidth,
            index: number,
            imageRows: IGroupedImagesWithWidth[],
        ): React.ReactNode => {
            const isLastRow = index === imageRows.length - 1;
            const isNarrow =
                isLastRow && imageRow.totalProportion < MIN_PROPORTIONS_SUM;
            const rowWidth = isNarrow
                ? `${(100 / MIN_PROPORTIONS_SUM) * imageRow.totalProportion}%`
                : undefined;

            return (
                <div
                    className={cx('imageRow', {imageRow_type_narrow: isNarrow})}
                    style={{width: rowWidth}}
                    key={index}
                >
                    {imageRow.items.map(renderImage)}
                </div>
            );
        };

        const renderPlaceholder = (): React.ReactNode => (
            <div className={cx('placeholder')}>
                <Spinner />
            </div>
        );

        return (
            <>
                <Modal
                    isVisible={isVisible && !isViewerVisible}
                    className={cx('root')}
                    containerClassName={cx('content')}
                    dialogClassName={cx('dialog')}
                    onClose={onCloseClick}
                >
                    <HotelImagesTitle className={cx('title')} hotel={hotel} />
                    <div className={cx('images')} ref={scrollableRef}>
                        {groupedImages.length > 0 && (
                            <LazyLoad
                                isLoading={isLoading}
                                placeholdersCount={1}
                                placeholderNode={renderPlaceholder()}
                                onBottomScroll={handleMoreImages}
                            >
                                {groupedImages.map(renderRow)}
                            </LazyLoad>
                        )}
                    </div>
                </Modal>
                <HotelPageImagesViewer
                    isVisible={isVisible && isViewerVisible}
                    totalImageCount={totalImageCount}
                    name={hotelName}
                    images={images}
                    initialIndex={viewerImageIndex}
                    onMoreImages={handleMoreImages}
                    onCloseClick={handleCloseClick}
                />
            </>
        );
    };

export default HotelImagesDesktop;
