import {
    useMemo,
    useState,
    useCallback,
    FunctionComponent,
    ReactNode,
} from 'react';

import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {IHotel} from 'types/hotels/hotel/IHotel';

import {reachGoal} from 'utilities/metrika';
import {convertUrlTemplateImagesToFlatList} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import * as i18nBlockCommon from 'i18n/common';

/* Components */
import ModalWithHistoryBack from 'containers/withSupportHistoryBack/ModalWithHistoryBack/ModalWithHistoryBack';

import Image from 'components/TravelImage/TravelImage';
import LazyLoad from 'projects/depreacted/hotels/components/LazyLoad/LazyLoad';
import HotelPageImagesViewer from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageImagesViewer/HotelPageImagesViewer';
import LinkButton from 'components/LinkButton/LinkButton';
import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import ArrowLeftIcon from 'icons/16/ArrowLeft';

import HotelImagesImage from '../HotelImagesImage/HotelImagesImage';
import HotelImagesTitle from '../HotelImagesTitle/HotelImagesTitle';

/* Init styles */

import cx from './HotelImagesMobile.scss';

const ITEMS_TO_LOAD = 10;
const HOTEL_IMAGE_MOBILE_SIZE_NAME = 'M';

/* Component Types */
interface IHotelImagesMobileProps {
    images: IHotelOrRoomImage[];
    totalImageCount: number;
    hotel: IHotel;
    isLoading: boolean;
    isVisible: boolean;
    onMoreImages: (count: number) => void;
    onClose: () => void;
}

const HotelImagesMobile: FunctionComponent<IHotelImagesMobileProps> = ({
    images,
    hotel,
    totalImageCount,
    isLoading,
    isVisible,
    onMoreImages,
    onClose,
}) => {
    const {name: hotelName} = hotel;
    const hasImages = images.length > 0;

    const [isViewerVisible, setViewerVisible] = useState(false);
    const [viewerImageIndex, setViewerIndex] = useState<number>();

    const loadImageCount = useMemo(() => {
        return Math.min(ITEMS_TO_LOAD, totalImageCount - images.length);
    }, [images, totalImageCount]);

    const convertedImages = useMemo(
        () =>
            convertUrlTemplateImagesToFlatList(
                images,
                HOTEL_IMAGE_MOBILE_SIZE_NAME,
            ),
        [images],
    );

    const handleImageClick = useCallback(
        (index: number) => {
            setViewerIndex(index);
            setViewerVisible(true);

            reachGoal(EHotelsGoal.HOTEL_PAGE_IMAGES_TAB_IMAGE_CLICK);
        },
        [setViewerIndex, setViewerVisible],
    );

    const handleCloseClick = useCallback(() => {
        setViewerVisible(false);
    }, [setViewerVisible]);

    const handleMoreImages = useCallback(() => {
        onMoreImages(loadImageCount);
    }, [onMoreImages, loadImageCount]);

    const renderPlaceholder = (): ReactNode => (
        <Image className={cx('imagePlaceholder')} src="" isPlaceholder />
    );

    const renderImage = (src: string, index: number): ReactNode => {
        return (
            <HotelImagesImage
                src={src}
                key={src + index}
                imageIndex={index}
                onImageClick={handleImageClick}
                className={cx('imageWrap')}
                imageClassName={cx('image')}
            />
        );
    };

    return (
        <>
            <ModalWithHistoryBack
                isVisible={isVisible}
                className={cx('root')}
                containerClassName={cx('content')}
                onClose={onClose}
                onHistoryBack={onClose}
                hasCloseButton={false}
                resetScrollOnOpen
            >
                <LinkButton onClick={onClose} className={cx('back')}>
                    <TextWithIcon
                        iconLeft={ArrowLeftIcon}
                        text={i18nBlockCommon.back()}
                    />
                </LinkButton>
                <HotelImagesTitle className={cx('title')} hotel={hotel} />
                <div className={cx('images')}>
                    {hasImages && (
                        <LazyLoad
                            isLoading={isLoading}
                            placeholdersCount={loadImageCount}
                            placeholderNode={renderPlaceholder()}
                            onBottomScroll={handleMoreImages}
                        >
                            {convertedImages.map(renderImage)}
                        </LazyLoad>
                    )}
                </div>
            </ModalWithHistoryBack>
            <HotelPageImagesViewer
                isVisible={isViewerVisible}
                totalImageCount={totalImageCount}
                name={hotelName}
                images={images}
                initialIndex={viewerImageIndex}
                onMoreImages={handleMoreImages}
                onCloseClick={handleCloseClick}
            />
        </>
    );
};

export default HotelImagesMobile;
