import React, {createRef, PureComponent, ReactNode} from 'react';
import {RouteComponentProps} from 'react-router';
import _last from 'lodash/last';

import {EHotelPageModalStatus} from 'projects/depreacted/hotels/constants/hotelPage';
import {URLs} from 'constants/urls';

import {IWithClassName} from 'types/withClassName';
import {PermalinkType} from 'types/hotels/hotel/IHotel';
import {
    EHotelBanner,
    IHotelInfo,
} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';
import {EAdFoxBannerPosition, EAdFoxBannerType} from 'types/AdFox';

import {IDevice} from 'reducers/common/commonReducerTypes';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {hotelsURLs} from 'projects/depreacted/hotels/utilities/urls';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import debounceById from 'utilities/functions/debounceById';
import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nFavoritesBlock from 'i18n/account-Favorites';

import {IHotelPageCardContainer} from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageCard/HotelPageCardContainer';
import HotelPageCardMainTab from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageCardMainTab/HotelPageCardMainTab';
import HotelPageCardHeaderRedesign from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageCardHeader/HotelPageCardHeader';
import Modal, {IModalProps} from 'components/Modal/Modal';
import MobileHotelPageRoomDetailedInfo from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageAllAmenities/blocks/MobileHotelPageRoomDetailedInfo/MobileHotelPageRoomDetailedInfo';
import HotelPageAllAmenities from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageAllAmenities/HotelPageAllAmenities';
import HotelPageCardMap from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageCardMap/HotelPageCardMap';
import BottomSheet, {
    IBottomSheetProps,
} from 'components/BottomSheet/BottomSheet';
import LinkButton from 'components/LinkButton/LinkButton';
import Snackbar from 'projects/depreacted/hotels/components/Snackbar/Snackbar';
import CloseForCircleBgIcon from 'icons/24/CloseForCircleBg';
import Button from 'components/Button/Button';
import HotelOfferBanner from 'projects/depreacted/hotels/components/HotelOfferBanner/HotelOfferBanner';
import AdFoxBanner from 'components/AdFoxBanner/AdFoxBanner';

import cx from './HotelPageCard.scss';

/* Component Types */
interface IHotelPageCardProps extends IHotelPageCardContainer, IWithClassName {
    deviceType: IDevice;
    location: TReactRouterLocation;
    hotelInfo: IHotelInfo;
    isLoadingOffers: boolean;
    isHeadless: boolean;
}

interface IHotelPageCardState {
    modalStatus?: EHotelPageModalStatus;
    canPrerender?: boolean;
    roomId?: string;
}

type TReactRouterLocation = RouteComponentProps['location'];

/* Constants */
const HOTEL_PAGE_QA = 'hotels-hotelPage';
const MAP_TAB_QA = 'hotelMapTab';
const SET_ON_FAVORITES_CLICK_DEBOUNCE = 2000;

class HotelPageCard extends PureComponent<
    IHotelPageCardProps,
    IHotelPageCardState
> {
    reviewsRef = createRef<HTMLDivElement>();

    constructor(props: IHotelPageCardProps) {
        super(props);

        this.state = {
            modalStatus: EHotelPageModalStatus.closed,
        };
    }

    state: IHotelPageCardState;

    componentDidMount(): void {
        this.setState({canPrerender: true});
    }

    /* Handlers */

    private handleReviewsClick = (): void => {
        if (this.reviewsRef.current) {
            const {deviceType} = this.props;
            const headerHeight = deviceType.isMobile ? 0 : 80;
            const y =
                this.reviewsRef.current.getBoundingClientRect().top +
                window.pageYOffset -
                headerHeight;

            window.scrollTo({top: y, behavior: 'smooth'});

            reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_REVIEWS_COUNT_CLICK);
        }
    };

    private handleStaticMapClick = (): void => {
        reachGoal(EHotelsGoal.HOTELS_HOTEL_PAGE_STATIC_MAP_CLICK);

        this.setState({
            modalStatus: EHotelPageModalStatus.map,
        });
    };

    private handleOffersButtonClick = (): void => {
        this.setState({roomId: undefined});
    };

    private handleOfferWatchButtonClick = (roomId: string): void => {
        this.setState({
            roomId,
            modalStatus: EHotelPageModalStatus.allAmenities,
        });
    };

    private handleReviewsButtonClick = (): void => {
        // this.setState({
        //     modalStatus: EHotelPageModalStatus.reviews,
        // });
    };

    /* Actions */

    private closeModalOrBottomSheet = (): void => {
        this.setState({modalStatus: EHotelPageModalStatus.closed});
    };

    private getBackButtonLink = (): string | undefined => {
        const {
            hotelInfo: {
                breadcrumbs: {geoRegions},
            },
            location,
        } = this.props;

        const region = _last(geoRegions);

        return region
            ? hotelsURLs.getHotelsSearchUrl(location, region.geoId)
            : undefined;
    };

    /* Render */
    renderCardHeader(): ReactNode {
        const {
            hotelInfo: {
                hotel,
                ratingsInfo,
                breadcrumbs: {geoRegions},
                offersInfo: {bannerType},
                hotel: {name},
                searchParams,
            },
            deviceType,
            changeHotelsIsFavoriteAction,
            removeFavoriteHotel,
            addFavoriteHotel,
            setSnackbarInfo,
        } = this.props;

        const region = _last(geoRegions);

        const canRenderHotelOfferBanner =
            !bannerType || bannerType !== EHotelBanner.NONE;

        const debouncedSearchReactionRequest = debounceById(
            (permalink: string) => {
                changeHotelsIsFavoriteAction({permalink});
            },
            SET_ON_FAVORITES_CLICK_DEBOUNCE,
        );

        const handleFavoritesClick = ({
            permalink,
            isFavorite,
        }: {
            permalink: PermalinkType;
            isFavorite: boolean;
        }): void => {
            if (isFavorite) {
                removeFavoriteHotel({permalink});
                setSnackbarInfo({
                    lastAction: `${permalink}-remove`,
                    hasCancellation: true,
                    message: i18nFavoritesBlock.removeFavoriteHotel(),
                    cancelAction: 'addFavoriteHotelOnHotelPage',
                    cancelActionData: {requestParams: {permalink}},
                    page: 'hotel',
                });
            } else {
                addFavoriteHotel({
                    requestParams: {permalink},
                    message: insertJSXIntoKey(
                        i18nFavoritesBlock.addFavoriteHotel,
                    )({
                        favorites: (
                            <LinkButton href={URLs.favorites} theme="brand">
                                {i18nFavoritesBlock.favorites()}
                            </LinkButton>
                        ),
                    }),
                });
            }

            debouncedSearchReactionRequest(permalink, permalink);
        };

        return (
            <>
                {canRenderHotelOfferBanner ? (
                    <HotelOfferBanner
                        className={cx(
                            'banner',
                            deviceModMobile('banner', deviceType),
                        )}
                        mirBannerClassName={cx('mirBanner')}
                        bannerType={bannerType}
                        hotelName={name}
                        {...prepareQaAttributes(HOTEL_PAGE_QA)}
                    />
                ) : (
                    <AdFoxBanner
                        className={cx(
                            'banner',
                            deviceModMobile('banner', deviceType),
                        )}
                        type={EAdFoxBannerType.Inline}
                        position={EAdFoxBannerPosition.Center}
                        fixed
                    />
                )}

                <HotelPageCardHeaderRedesign
                    className={cx('header')}
                    hotel={hotel}
                    ratingsInfo={ratingsInfo}
                    withoutDates={!searchParams}
                    backGeoRegionLink={this.getBackButtonLink()}
                    backGeoRegion={region}
                    onHotelAddressClick={this.handleStaticMapClick}
                    deviceType={deviceType}
                    onFavoritesClick={handleFavoritesClick}
                    onReviewsClick={this.handleReviewsClick}
                    {...prepareQaAttributes(HOTEL_PAGE_QA)}
                />
            </>
        );
    }

    renderHotelCardMainTab(): ReactNode {
        const {
            deviceType,
            hotelInfo,
            isLoadingOffers,
            getHotelImages,
            hotelImages,
            experiments: {roomsMatching},
        } = this.props;

        return (
            <HotelPageCardMainTab
                deviceType={deviceType}
                isLoadingOffers={isLoadingOffers}
                onOffersButtonClick={this.handleOffersButtonClick}
                handleOfferWatchButtonClick={this.handleOfferWatchButtonClick}
                onReviewsButtonClick={this.handleReviewsButtonClick}
                onStaticMapClick={this.handleStaticMapClick}
                backButtonLink={this.getBackButtonLink()}
                hotelInfo={hotelInfo}
                onHotelAddressClick={this.handleStaticMapClick}
                getHotelImages={getHotelImages}
                hotelImages={hotelImages}
                reviewsRef={this.reviewsRef}
                withRoomsMatching={roomsMatching}
                {...prepareQaAttributes(HOTEL_PAGE_QA)}
            />
        );
    }

    renderContent(): ReactNode {
        const {isHeadless, deviceType} = this.props;

        if (isHeadless) {
            return this.renderHotelCardMainTab();
        }

        return (
            <>
                {this.renderCardHeader()}
                {this.renderHotelCardMainTab()}
                <Snackbar deviceType={deviceType} currentPage="hotel" />
            </>
        );
    }

    wrapModalContentIfNeed(content: ReactNode): ReactNode {
        const {modalStatus} = this.state;

        if (modalStatus === EHotelPageModalStatus.map) {
            return content;
        }

        return <Modal.Content>{content}</Modal.Content>;
    }

    renderModalOrBottomSheet(): ReactNode {
        const {deviceType} = this.props;
        const {isMobile} = deviceType;
        const content = this.renderModalOrBottomSheetContent();

        if (isMobile) {
            return (
                <BottomSheet {...this.getBottomSheetProps()}>
                    {content}
                </BottomSheet>
            );
        }

        return (
            <Modal {...this.getModalProps()}>
                {this.wrapModalContentIfNeed(content)}
            </Modal>
        );
    }

    getModalProps(): IModalProps {
        const {modalStatus} = this.state;
        const commonProps: Partial<IModalProps> = {
            onClose: this.closeModalOrBottomSheet,
            disableAnimations: true,
        };

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                return {
                    ...commonProps,
                    isVisible:
                        modalStatus === EHotelPageModalStatus.allAmenities,
                    className: cx('modal_allAmenities'),
                    containerClassName: cx('modal_allAmenitiesContainer'),
                };
            }

            case EHotelPageModalStatus.map: {
                return {
                    ...commonProps,
                    isVisible: modalStatus === EHotelPageModalStatus.map,
                    containerClassName: cx('modalContainerMap'),
                    renderCloseButton: (close): React.ReactNode => (
                        <Button
                            className={cx('closeMapButton')}
                            shape="circle"
                            theme="raised"
                            onClick={close}
                        >
                            <CloseForCircleBgIcon />
                        </Button>
                    ),
                };
            }

            default: {
                return {
                    ...commonProps,
                    isVisible: false,
                };
            }
        }
    }

    getBottomSheetProps(): IBottomSheetProps {
        const {modalStatus} = this.state;
        const commonProps = {
            onClose: this.closeModalOrBottomSheet,
        };

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                return {
                    ...commonProps,
                    isOpened:
                        modalStatus === EHotelPageModalStatus.allAmenities,
                };
            }

            case EHotelPageModalStatus.map: {
                return {
                    ...commonProps,
                    isOpened: modalStatus === EHotelPageModalStatus.map,
                    contentClassName: cx('bottomSheet_map'),
                    dragDisabledContent: true,
                };
            }

            default: {
                return {
                    ...commonProps,
                    isOpened: false,
                };
            }
        }
    }

    renderModalOrBottomSheetContent(): ReactNode {
        const {
            deviceType,
            hotelInfo: {
                hotel,
                offersInfo: {rooms},
            },
        } = this.props;
        const {isMobile} = deviceType;
        const {modalStatus, roomId} = this.state;

        switch (modalStatus) {
            case EHotelPageModalStatus.allAmenities: {
                if (roomId === undefined) {
                    return null;
                }

                const room = rooms?.find(
                    currentRoom => currentRoom.id === roomId,
                );

                if (!room) {
                    return null;
                }

                if (isMobile) {
                    return <MobileHotelPageRoomDetailedInfo room={room} />;
                }

                return <HotelPageAllAmenities room={room} />;
            }

            case EHotelPageModalStatus.map: {
                return (
                    <HotelPageCardMap
                        className={cx(deviceMods('map', deviceType))}
                        hotel={hotel}
                        canRenderAddress={isMobile}
                        deviceType={deviceType}
                        {...prepareQaAttributes(MAP_TAB_QA)}
                    />
                );
            }

            default: {
                return null;
            }
        }
    }

    render(): ReactNode {
        const {className, deviceType} = this.props;

        return (
            <div
                className={cx(deviceMods('root', deviceType), className)}
                {...prepareQaAttributes('hotelCard')}
            >
                {this.renderContent()}
                {this.renderModalOrBottomSheet()}
            </div>
        );
    }
}

export default HotelPageCard;
