import React, {PureComponent, ReactElement, ReactNode} from 'react';
import noop from 'lodash/noop';
import {withRouter, RouteComponentProps} from 'react-router-dom';

import {IGeoRegion} from 'types/hotels/hotel/IGeoRegion';
import {IHotel, PermalinkType} from 'types/hotels/hotel/IHotel';
import {IWithDeviceType} from 'types/withDeviceType';
import {IWithClassName} from 'types/withClassName';
import {IRatingsInfo} from 'server/api/HotelSearchAPI/types/IHotelInfo';
import {EFavoritesGoal} from 'utilities/metrika/types/goals/favorites';
import EPopupDirection from 'components/Popup/types/EPopupDirection';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';

import * as i18nBlockPage from 'i18n/hotels-HotelPage';

import HotelAddress from 'projects/depreacted/hotels/components/HotelAddress/HotelAddress';
import Link from 'components/Link/Link';
import ArrowLeftIcon from 'icons/16/ArrowLeft';
import TransportAccessibility from 'projects/depreacted/hotels/components/TransportAccessibility/TransportAccessibility';
import HotelRatingWithReviewLink from 'projects/depreacted/hotels/components/HotelRatingWithReviewLink/HotelRatingWithReviewLink';
import Dropdown, {IDropdownSwitcherParams} from 'components/Dropdown/Dropdown';
import HotelFeatures from 'projects/depreacted/hotels/components/HotelReviews/components/HotelFeatures/HotelFeatures';
import HotelTitle from 'projects/depreacted/hotels/pages/HotelPage/components/HotelTitle/HotelTitle';
import Location from 'icons/16/Location';
import HotelGeoFeature from 'projects/depreacted/hotels/components/HotelGeoFeature/HotelGeoFeature';
import Box from 'components/Box/Box';

import cx from './HotelPageCardHeader.scss';

interface IHotelInfoProps
    extends IWithClassName,
        IWithDeviceType,
        RouteComponentProps,
        IWithQaAttributes {
    hotel: IHotel;
    ratingsInfo: IRatingsInfo;
    withoutDates: boolean;
    backGeoRegionLink?: string;
    backGeoRegion?: IGeoRegion;
    onTotalReviewLinkClick?: () => void;
    onReviewsClick: () => void;
    onHotelAddressClick?: () => void;
    onFavoritesClick?: ({
        permalink,
        isFavorite,
    }: {
        permalink: PermalinkType;
        isFavorite: boolean;
    }) => void;
}

class HotelPageCardHeader extends PureComponent<IHotelInfoProps> {
    private getButtonLinkProps() {
        const {history, backGeoRegionLink} = this.props;

        if (history.location.state && history.location.state.goBack) {
            return {
                to: undefined,
                onClick: history.goBack,
            };
        }

        return {
            to: backGeoRegionLink,
        };
    }

    /* Render */
    private renderBackButton(): ReactNode {
        const {backGeoRegionLink, backGeoRegion, deviceType} = this.props;

        const geoRegionGevitiveCase = backGeoRegion
            ? backGeoRegion.linguistics.genitiveCase
            : undefined;

        if (backGeoRegionLink && geoRegionGevitiveCase) {
            return (
                // @ts-ignore
                <Link
                    className={cx(
                        'backButton',
                        deviceMods('backButton', deviceType),
                    )}
                    {...this.getButtonLinkProps()}
                    {...prepareQaAttributes({
                        current: 'backButton',
                        parent: this.props,
                    })}
                >
                    <ArrowLeftIcon className={cx('backButtonArrow')} />
                    {`${i18nBlockPage.geoRegionPrefix()} ${geoRegionGevitiveCase}`}
                </Link>
            );
        }

        return null;
    }

    private renderHotelName(): ReactNode {
        const {
            hotel: {name, stars, isFavorite, permalink},
            deviceType,
            onFavoritesClick,
        } = this.props;

        const handleFavoriteClick = () => {
            if (onFavoritesClick) {
                onFavoritesClick({permalink, isFavorite});
            }

            reachGoal(EFavoritesGoal.FAVORITES_CLICK);
        };

        return (
            <HotelTitle
                name={name}
                stars={stars}
                isFavorite={isFavorite}
                handleFavoriteClick={handleFavoriteClick}
                deviceType={deviceType}
            />
        );
    }

    private renderHotelAddress(): ReactNode {
        const {
            hotel: {address},
            onHotelAddressClick = noop,
        } = this.props;

        return (
            <HotelAddress
                className={cx('address')}
                text={address}
                type="link"
                onClick={onHotelAddressClick}
                icon={Location}
                multiLine
            />
        );
    }

    private renderGeoFeature(): ReactNode {
        const {
            hotel: {geoFeature},
        } = this.props;

        return (
            <HotelGeoFeature
                geoFeature={geoFeature}
                {...prepareQaAttributes({
                    current: 'geoFeature',
                    parent: this.props,
                })}
            />
        );
    }

    private renderTransportAccessibility(): ReactNode {
        const {
            hotel: {nearestStations},
        } = this.props;

        return (
            nearestStations?.[0] && (
                <TransportAccessibility
                    textClassName={cx('transportAccessibility_text')}
                    bigIcon
                    station={nearestStations[0]}
                    {...prepareQaAttributes({
                        current: 'transportAccessibility',
                        parent: this.props,
                    })}
                />
            )
        );
    }

    private renderDesktopRightColumnPopup = () => {
        return <HotelFeatures />;
    };

    private renderDesktopRightColumn = (): React.ReactElement => {
        return (
            <Dropdown
                renderSwitcher={this.renderDesktopRightColumnSwitcher}
                renderPopup={this.renderDesktopRightColumnPopup}
                popupProps={{
                    directions: [EPopupDirection.BOTTOM_RIGHT],
                    className: cx('rightColumn_popup'),
                }}
            />
        );
    };

    private renderDesktopRightColumnSwitcher = ({
        input: {ref},
        meta: {showPopup, hidePopup},
    }: IDropdownSwitcherParams<unknown>): React.ReactElement => {
        const {
            onReviewsClick = noop,
            ratingsInfo: {featureRatings},
        } = this.props;
        const canRenderFeatures = featureRatings && featureRatings.length > 0;
        const handlers = canRenderFeatures
            ? {
                  onMouseEnter: showPopup,
                  onMouseLeave: hidePopup,
              }
            : {};

        return (
            <div
                ref={ref}
                className={cx('rightColumn')}
                onClick={onReviewsClick}
                {...handlers}
            >
                {this.renderHotelRating()}
                {this.renderPositiveReview()}
            </div>
        );
    };

    private renderHotelRating = (): ReactElement => {
        const {
            hotel: {rating, totalTextReviewCount},
            deviceType: {isMobile},
            onReviewsClick = noop,
        } = this.props;

        return (
            <HotelRatingWithReviewLink
                className={cx('hotelRating')}
                rating={rating}
                totalTextReviewCount={totalTextReviewCount}
                onTotalReviewLinkClick={isMobile ? onReviewsClick : undefined}
                isMobile={isMobile}
            />
        );
    };

    private renderPositiveReview(): ReactNode {
        const {ratingsInfo} = this.props;

        if (ratingsInfo && ratingsInfo.teaser) {
            return (
                <div
                    className={cx('positiveReview')}
                    {...prepareQaAttributes('positiveReview')}
                >
                    {ratingsInfo.teaser}
                </div>
            );
        }

        return null;
    }

    render(): ReactNode {
        const {deviceType, className} = this.props;
        const {isMobile, isDesktop} = deviceType;

        return (
            <>
                {isDesktop && this.renderBackButton()}
                <div className={cx(deviceMods('root', deviceType), className)}>
                    <div className={cx('leftColumn')}>
                        {isMobile && this.renderBackButton()}
                        {this.renderHotelName()}
                        {isDesktop && (
                            <Box between={4} inline>
                                {this.renderHotelAddress()}
                                {this.renderGeoFeature()}
                                {this.renderTransportAccessibility()}
                            </Box>
                        )}
                    </div>
                    {isMobile
                        ? this.renderHotelRating()
                        : this.renderDesktopRightColumn()}
                </div>
            </>
        );
    }
}

export default withRouter(HotelPageCardHeader);
