import React, {createRef, PureComponent, ReactNode} from 'react';

import {
    RIGHT_COLUMN_OFFSET,
    RIGHT_COLUMN_WIDTH,
} from 'projects/depreacted/hotels/constants/hotelPage';

import {IWithClassName} from 'types/withClassName';
import {IHotelInfo} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';
import {EHotelsGoal} from 'utilities/metrika/types/goals/hotels';

import {IDevice} from 'reducers/common/commonReducerTypes';
import {GetHotelImagesActionType} from 'reducers/depreacted/hotels/hotelPage/hotelImages/actions';

import {HotelImagesSelectorType} from 'selectors/depreacted/hotels/hotel/imagesTab/getHotelImages';

import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceModDesktop} from 'utilities/stylesUtils';
import {reachGoal} from 'utilities/metrika';
import checkOffersDiscount from '../../utilities/checkOffersDiscount';

import * as i18nBlock from 'i18n/hotels-HotelPage';

import HotelAddress from 'projects/depreacted/hotels/components/HotelAddress/HotelAddress';
import {TwoColumnLayout} from 'components/Layouts/TwoColumnLayout/TwoColumnLayout';
import SimilarHotels from 'projects/depreacted/hotels/components/SimilarHotels/SimilarHotels';
import HotelReviewsRedesign from 'projects/depreacted/hotels/components/HotelReviews/HotelReviewsContainerRedesign';
import Separator from 'components/Separator/Separator';
import Price from 'components/Price/Price';
import Button from 'components/Button/Button';
import Heading from 'components/Heading/Heading';
import TransportAccessibility from 'projects/depreacted/hotels/components/TransportAccessibility/TransportAccessibility';
import GroupedAmenities from 'projects/depreacted/hotels/components/Amenities/GroupedAmenities';
import OffersInfo from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/OffersInfo';
import HotelsGalleryRedesign from 'projects/depreacted/hotels/pages/HotelPage/components/GalleryRedesign/GalleryRedesign';
import HotelFeatures from 'projects/depreacted/hotels/components/HotelReviews/components/HotelFeatures/HotelFeatures';
import HotelPageGeoInfo from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageGeoInfo/HotelPageGeoInfo';
import HotelPageBudapeshtCards from 'projects/depreacted/hotels/components/HotelPageBudapeshtCards/HotelPageBudapeshtCards';
import BorderPositioned from 'components/BorderPositioned/BorderPositioned';
import SellIcon from 'icons/24/Sell';
import Location from 'icons/16/Location';
import Flex from 'components/Flex/Flex';
import HotelGeoFeature from 'projects/depreacted/hotels/components/HotelGeoFeature/HotelGeoFeature';
import HotelImages from 'projects/depreacted/hotels/pages/HotelPage/components/HotelImages/HotelImagesContainer';

import HotelPageMobileMainImage from './blocks/HotelPageMobileMainImage/HotelPageMobileMainImage';
import HotelPageAmenities from './blocks/HotelPageAmenities/HotelPageAmenities';

import cx from './HotelPageCardMainTab.scss';

/* Component Types */
interface IHotelPageCardMainTabProps extends IWithClassName, IWithQaAttributes {
    hotelInfo: IHotelInfo;
    isLoadingOffers: boolean;
    onOffersButtonClick: () => void;
    handleOfferWatchButtonClick: (roomId: string) => void;
    onReviewsButtonClick: () => void;
    onHotelAddressClick: () => void;
    onStaticMapClick: () => void;
    backButtonLink?: string;
    deviceType: IDevice;
    getHotelImages: GetHotelImagesActionType;
    hotelImages: HotelImagesSelectorType;
    reviewsRef: React.RefObject<HTMLDivElement>;
    withRoomsMatching?: boolean;
}
interface IHotelPageCardMainTabState {
    isAllAmenities: boolean;
    canRenderModalAmenities: boolean;
    isModalImagesViewerVisible: boolean;
    viewerImageIndex: number;
    isImagesLoaded: boolean;
}

const ITEMS_TO_LOAD_MOBILE = 10;
const ITEMS_TO_LOAD_DESKTOP = 16;
const MAX_ITEMS_LEFT = 20;
const DEFAULT_IMAGE_LIMIT = 20;
const ABOUT_HOTEL_QA = 'aboutHotel';

class HotelPageCardMainTab extends PureComponent<
    IHotelPageCardMainTabProps,
    IHotelPageCardMainTabState
> {
    state: IHotelPageCardMainTabState = {
        isAllAmenities: false,
        canRenderModalAmenities: false,
        isModalImagesViewerVisible: false,
        viewerImageIndex: 0,
        isImagesLoaded: false,
    };

    roomsRef = createRef<HTMLDivElement>();

    componentDidMount(): void {
        this.loadHotelImagesIfNeed();
    }

    componentDidUpdate(): void {
        this.loadHotelImagesIfNeed();
    }

    /* Actions */

    loadHotelImagesIfNeed(): void {
        const {
            hotelImages: {data, isLoading, isSuccess},
            getHotelImages,
        } = this.props;
        const {isImagesLoaded} = this.state;

        if (
            !isImagesLoaded &&
            !isSuccess &&
            !isLoading &&
            data.images.length === 0
        ) {
            this.setState(
                {
                    isImagesLoaded: true,
                },
                () =>
                    getHotelImages({
                        imageOffset: data.images.length,
                        imageLimit: DEFAULT_IMAGE_LIMIT,
                    }),
            );
        }
    }

    /* Handlers */

    private handleImagesClick = (index: number) => {
        reachGoal(EHotelsGoal.HOTEL_PAGE_IMAGES_TAB_IMAGE_CLICK);
        this.setState({
            isModalImagesViewerVisible: true,
            viewerImageIndex: index,
        });
    };

    /* Helpers */

    loadImagesByCount = (itemsCount: number) => () => {
        const {
            hotelImages: {isLoading, data},
            getHotelImages,
        } = this.props;

        if (!isLoading && data.images.length < data.totalImageCount) {
            getHotelImages({
                imageOffset: data.images.length,
                imageLimit: itemsCount,
            });
        }
    };

    getLoadImageCount(): number {
        const {
            deviceType,
            hotelImages: {
                data: {images, totalImageCount},
            },
        } = this.props;

        if (deviceType.isMobile) {
            return Math.min(
                ITEMS_TO_LOAD_MOBILE,
                totalImageCount - images.length,
            );
        }

        if (totalImageCount - images.length < MAX_ITEMS_LEFT) {
            return totalImageCount - images.length;
        }

        return ITEMS_TO_LOAD_DESKTOP;
    }

    /* Handlers */

    private handleClickSimilarHotel = (): void => {
        reachGoal(EHotelsGoal.HOTEL_PAGE_MAIN_TAB_SIMILAR_HOTEL_CLICK);
    };

    private handleOfferSelect = (): void => {
        reachGoal(EHotelsGoal.HOTEL_PAGE_MAIN_TAB_OFFER_CLICK);
    };

    private handleOfferWatchButtonClick = (roomId: string): void => {
        this.props.handleOfferWatchButtonClick(roomId);
    };

    private handleShowAmenities = (): void => {
        reachGoal(EHotelsGoal.HOTEL_PAGE_MAIN_TAB_SHOW_AMENITIES);
    };

    private handleImagesCloseClick = (): void => {
        this.setState({isModalImagesViewerVisible: false});
    };

    private scrollToOffersInfo = () => {
        if (this.roomsRef.current) {
            this.roomsRef.current.scrollIntoView();
        }
    };

    /* Render */

    private renderMobileMainImage(): ReactNode {
        const {
            hotelInfo: {
                hotel: {images},
            },
            hotelImages,
        } = this.props;

        const loadImageCount = this.getLoadImageCount();

        if (images.length) {
            return (
                <HotelPageMobileMainImage
                    className={cx('mainImage_mobile')}
                    images={images}
                    hotelImages={hotelImages}
                    onImagesClick={this.handleImagesClick}
                    loadMoreImages={this.loadImagesByCount(loadImageCount)}
                    {...prepareQaAttributes('hotelPageGallery')}
                />
            );
        }

        return null;
    }

    private renderImagesGallery(): ReactNode {
        const {
            hotelInfo: {
                hotel: {images, name, totalImageCount},
            },
            deviceType,
        } = this.props;

        if (!totalImageCount) {
            return (
                <Separator
                    className={cx(
                        deviceModDesktop('gallerySeparator', deviceType),
                    )}
                />
            );
        }

        return (
            <HotelsGalleryRedesign
                className={cx(deviceModDesktop('gallery', deviceType))}
                images={images}
                imagesAlt={name}
                totalImageCount={totalImageCount}
                onMoreImagesClick={this.handleImagesClick}
            />
        );
    }

    private renderAboutHotelInfo(): ReactNode {
        const {deviceType} = this.props;
        const {isMobile} = deviceType;

        const hotelDescriptionNode = this.renderHotelDescription();
        const amenitiesNode = this.renderAmenitiesNode();
        const budapeshtNode = !isMobile && this.renderBudapesht();

        if (hotelDescriptionNode || amenitiesNode) {
            return (
                <div
                    className={cx(
                        'aboutHotel',
                        deviceModDesktop('leftColumnShift', deviceType),
                    )}
                    {...prepareQaAttributes(ABOUT_HOTEL_QA)}
                >
                    {!isMobile && (
                        <Heading level={2} className={cx('aboutHotelLabel')}>
                            {i18nBlock.aboutHotelLabel()}
                        </Heading>
                    )}
                    {budapeshtNode}
                    {hotelDescriptionNode}
                    {amenitiesNode}
                </div>
            );
        }

        return null;
    }

    private renderAmenitiesNode(): ReactNode {
        const {
            hotelInfo: {
                hotel: {name, mainAmenities, amenityGroups},
            },
            deviceType,
        } = this.props;

        const hasAmenities =
            (mainAmenities && mainAmenities.length) ||
            (amenityGroups && amenityGroups.length);

        if (!hasAmenities) {
            return null;
        }

        const hasAdditionalAmenities = Boolean(
            amenityGroups && amenityGroups.length,
        );

        const showAmenitiesText = deviceType.isDesktop
            ? i18nBlock.showAmenities()
            : i18nBlock.toggleAmenitiesButton();

        return (
            <HotelPageAmenities
                hasAdditionalAmenities={hasAdditionalAmenities}
                backButtonText={name}
                onShowAmenities={this.handleShowAmenities}
                mobilePopupTitle={i18nBlock.amenitiesTitle()}
                showAmenitiesText={showAmenitiesText}
                hideAmenitiesText={i18nBlock.hideAmenities()}
                renderDesktopAmenities={(isAll: boolean): ReactNode => (
                    <GroupedAmenities
                        mainAmenities={mainAmenities}
                        amenityGroups={amenityGroups}
                        isAllAmenities={isAll}
                        deviceType={deviceType}
                    />
                )}
                renderMobileMainAmenities={(): ReactNode => (
                    <GroupedAmenities
                        mainAmenities={mainAmenities}
                        amenityGroups={amenityGroups}
                        isAllAmenities={false}
                        deviceType={deviceType}
                    />
                )}
                renderMobileModalAmenities={(): ReactNode => (
                    <GroupedAmenities
                        mainAmenities={mainAmenities}
                        amenityGroups={amenityGroups}
                        isAllAmenities={true}
                        deviceType={deviceType}
                    />
                )}
                deviceType={deviceType}
                {...prepareQaAttributes(ABOUT_HOTEL_QA)}
            />
        );
    }

    private renderHotelDescription(): ReactNode {
        const {
            hotelInfo: {hotelDescription},
        } = this.props;

        if (hotelDescription && hotelDescription.text) {
            return (
                <div className={cx('hotelDescription')}>
                    {hotelDescription.text}
                </div>
            );
        }

        return null;
    }

    private renderBudapesht(): ReactNode {
        const {
            deviceType,
            hotelInfo: {
                hotel: {isYandexHotel},
            },
        } = this.props;

        if (isYandexHotel) {
            return (
                <HotelPageBudapeshtCards
                    deviceType={deviceType}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'budapeshtBlock',
                    })}
                />
            );
        }

        return null;
    }

    private renderHotelOffersInfo(): ReactNode {
        const {
            hotelInfo: {offersInfo, searchParams},
            onOffersButtonClick,
            backButtonLink,
            deviceType,
            isLoadingOffers,
            withRoomsMatching,
        } = this.props;

        return (
            <OffersInfo
                className={cx('offersInfo')}
                mainOffersClassName={cx(
                    deviceModDesktop('leftColumnShift', deviceType),
                )}
                partnerOffersClassName={cx(
                    'partnerOffersInfo',
                    deviceModDesktop('leftColumnShift', deviceType),
                )}
                roomsRef={this.roomsRef}
                offersInfo={offersInfo}
                searchParams={searchParams}
                deviceType={deviceType}
                backButtonLink={backButtonLink}
                isFullOffersTab
                onFilterSelect={onOffersButtonClick}
                onAllOffersClick={onOffersButtonClick}
                onOfferSelect={this.handleOfferSelect}
                onOfferWatchButtonClick={this.handleOfferWatchButtonClick}
                isLoading={isLoadingOffers}
                withRoomsMatching={withRoomsMatching}
                {...prepareQaAttributes('hotelOffersInfo')}
            />
        );
    }

    private renderHotelGeoInfo(): ReactNode {
        const {
            hotelInfo: {hotel},
            onStaticMapClick,
        } = this.props;

        return <HotelPageGeoInfo hotel={hotel} onMapClick={onStaticMapClick} />;
    }

    private renderSimilarHotels(): ReactNode {
        const {
            hotelInfo: {similarHotelsInfo, searchParams},
            deviceType,
        } = this.props;

        if (
            similarHotelsInfo &&
            similarHotelsInfo.hotels &&
            similarHotelsInfo.hotels.length > 0
        ) {
            return (
                <SimilarHotels
                    className={cx('similarHotels')}
                    similarHotelsInfo={similarHotelsInfo}
                    onClickCard={this.handleClickSimilarHotel}
                    searchParams={searchParams}
                    deviceType={deviceType}
                />
            );
        }

        return null;
    }

    private renderHotelReviews(): ReactNode {
        const {
            hotelInfo: {reviewsInfo},
            onReviewsButtonClick,
            deviceType,
            reviewsRef,
        } = this.props;

        if (reviewsInfo) {
            return (
                <HotelReviewsRedesign
                    reviewsRef={reviewsRef}
                    className={cx(
                        'reviews',
                        deviceModDesktop('leftColumnShift', deviceType),
                    )}
                    isActiveTab={false}
                    onMoreReviews={onReviewsButtonClick}
                    onKeyPhraseClick={onReviewsButtonClick}
                    deviceType={deviceType}
                />
            );
        }

        return null;
    }

    private renderHotelFeatures(): ReactNode {
        return <HotelFeatures className={cx('features')} />;
    }

    private renderOffersButton(): ReactNode {
        const {
            hotelInfo: {
                offersInfo: {aggregatedOfferInfo, mainOffers, partnerOffers},
            },
        } = this.props;

        const hasDiscount = checkOffersDiscount(mainOffers, partnerOffers);

        if (aggregatedOfferInfo?.minPrice) {
            const {minPrice} = aggregatedOfferInfo;

            return (
                <BorderPositioned
                    className={cx('viewAllOffers')}
                    badge={
                        hasDiscount && (
                            <SellIcon
                                className={cx('viewAllOffers_icon')}
                                width="24"
                                height="24"
                            />
                        )
                    }
                >
                    <Button
                        size="l"
                        theme="primary"
                        width="max"
                        onClick={this.scrollToOffersInfo}
                    >
                        {insertJSXIntoKey(i18nBlock.chooseRoomFromPrice)({
                            price: <Price key="viewAllOffers" {...minPrice} />,
                        })}
                    </Button>
                </BorderPositioned>
            );
        }

        return null;
    }

    private renderHotelAddress(): ReactNode {
        const {
            hotelInfo: {
                hotel: {address},
            },
            onHotelAddressClick,
            deviceType: {isMobile},
        } = this.props;

        return (
            <HotelAddress
                className={cx('address', {address_mobile: isMobile})}
                text={address}
                type="link"
                onClick={onHotelAddressClick}
                icon={Location}
                textClassName={cx('addressText')}
                multiLine
            />
        );
    }

    private renderGeoFeature(): ReactNode {
        const {
            hotelInfo: {
                hotel: {geoFeature},
            },
        } = this.props;

        return (
            <HotelGeoFeature
                geoFeature={geoFeature}
                {...prepareQaAttributes({
                    current: 'geoFeature',
                    parent: this.props,
                })}
            />
        );
    }

    private renderTransportAccessibility(): ReactNode {
        const {
            hotelInfo: {
                hotel: {nearestStations},
            },
        } = this.props;

        return (
            nearestStations?.[0] && (
                <TransportAccessibility
                    bigIcon
                    station={nearestStations[0]}
                    {...prepareQaAttributes({
                        current: 'transportAccessibility',
                        parent: this.props,
                    })}
                />
            )
        );
    }

    private renderMobileLayout(): ReactNode {
        return (
            <>
                {this.renderMobileMainImage()}
                <div className={cx('hotelAddressWrapper')}>
                    {this.renderOffersButton()}
                    {this.renderBudapesht()}
                    <Separator
                        className={cx(
                            'separatorBetweenOffersButtonAndAddress_mobile',
                        )}
                    />
                    {this.renderHotelAddress()}
                    <Flex
                        className={cx('featureAndTransportAccessibility')}
                        between={4}
                        inline
                    >
                        {this.renderGeoFeature()}
                        {this.renderTransportAccessibility()}
                    </Flex>
                    {this.renderAboutHotelInfo()}
                    <Separator margin={5} />
                    {this.renderHotelOffersInfo()}
                    {this.renderHotelFeatures()}
                    {this.renderHotelReviews()}
                    {this.renderSimilarHotels()}
                </div>
            </>
        );
    }

    private renderDesktopLayout(deviceType: IDevice): ReactNode {
        return (
            <TwoColumnLayout
                deviceType={deviceType}
                rightColumnOffset={RIGHT_COLUMN_OFFSET}
                rightColumnWidth={RIGHT_COLUMN_WIDTH}
            >
                <TwoColumnLayout.LeftColumn>
                    {this.renderAboutHotelInfo()}
                    <Separator margin={5} />
                    {this.renderHotelOffersInfo()}
                    {this.renderHotelReviews()}
                </TwoColumnLayout.LeftColumn>
                <TwoColumnLayout.RightColumn>
                    {this.renderHotelGeoInfo()}
                    {this.renderSimilarHotels()}
                    {this.renderHotelFeatures()}
                </TwoColumnLayout.RightColumn>
            </TwoColumnLayout>
        );
    }

    renderModalImagesViewer(): React.ReactNode {
        const {
            hotelInfo: {hotel},
        } = this.props;
        const {isModalImagesViewerVisible} = this.state;

        return (
            <HotelImages
                hotel={hotel}
                isVisible={isModalImagesViewerVisible}
                onCloseClick={this.handleImagesCloseClick}
            />
        );
    }

    render(): ReactNode {
        const {deviceType} = this.props;
        const {isDesktop, isMobile} = deviceType;

        return (
            <div
                className={cx({root_mobile: deviceType.isMobile})}
                {...prepareQaAttributes('hotelMainTab')}
            >
                {isDesktop && this.renderImagesGallery()}
                {isMobile
                    ? this.renderMobileLayout()
                    : this.renderDesktopLayout(deviceType)}
                {this.renderModalImagesViewer()}
            </div>
        );
    }
}

export default HotelPageCardMainTab;
