import {
    FunctionComponent,
    RefObject,
    ReactNode,
    useCallback,
    useMemo,
} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {
    IHotelOrRoomImage,
    IHotelOrRoomImageSize,
} from 'types/hotels/hotel/IHotelImages';

import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {deviceMods} from 'utilities/stylesUtils';
import {convertUrlTemplateImagesToFlatList} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';

import Carousel from 'components/Carousel/Carousel';
import Image from 'components/TravelImage/TravelImage';

import cx from './BottomCarousel.scss';

interface IBottomCarouselProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    images: IHotelOrRoomImage[];
    selectedIndex: number;
    totalImageCount: number;
    carouselRef: RefObject<Carousel>;
    imageSrcGetter?: (
        urlTemplate: string,
        size: string,
        availableSizes: IHotelOrRoomImageSize[],
    ) => string;
    onImageClick: (index: number) => void;
    onLastImageVisible?: () => void;
}

const DESKTOP_HEIGHT = 132;
const MOBILE_HEIGHT = 44;

const IMAGE_MOBILE_SIZE_NAME = 'M';
const IMAGE_SIZE_NAME = 'L';

const CAROUSEL_SCROLL_OFFSET = 250;
const CAROUSEL_SCROLL_OFFSET_MOBILE = 100;

const IMAGE_PLACEHOLDER = (
    <Image src="" key="placeholder" className={cx('image')} isPlaceholder />
);

const BottomCarousel: FunctionComponent<IBottomCarouselProps> = ({
    className,
    deviceType,
    images,
    selectedIndex,
    totalImageCount,
    carouselRef,
    imageSrcGetter,
    onImageClick,
    onLastImageVisible,
    ...props
}) => {
    const {isMobile} = deviceType;
    const carouselImageSize = isMobile
        ? IMAGE_MOBILE_SIZE_NAME
        : IMAGE_SIZE_NAME;

    const scrollOffset = isMobile
        ? CAROUSEL_SCROLL_OFFSET_MOBILE
        : CAROUSEL_SCROLL_OFFSET;

    const convertedImages = useMemo(
        () =>
            convertUrlTemplateImagesToFlatList(
                images,
                carouselImageSize,
                imageSrcGetter,
            ),
        [images, carouselImageSize, imageSrcGetter],
    );

    const renderImage = useCallback(
        (imgSrc: string, index: number): ReactNode => (
            <Image
                src={imgSrc}
                key={imgSrc}
                className={cx('image', {
                    image_selected: index === selectedIndex,
                })}
                onClick={(): void => onImageClick(index)}
                hasFitCover
                tagProps={index === selectedIndex ? {'data-active': true} : {}}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'carouselImage',
                })}
            />
        ),
        [onImageClick, selectedIndex, props],
    );

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), className)}
            {...prepareQaAttributes({parent: props, current: 'carousel'})}
        >
            <Carousel
                className={cx('carousel')}
                scroll="native"
                height={isMobile ? MOBILE_HEIGHT : DESKTOP_HEIGHT}
                spaceBetween={1}
                spaceBefore={1}
                spaceAfter={1}
                leftOffset={scrollOffset}
                rightOffset={scrollOffset}
                onLastItemVisible={onLastImageVisible}
                ref={carouselRef}
            >
                {convertedImages.map(renderImage)}
                {images.length < totalImageCount && IMAGE_PLACEHOLDER}
            </Carousel>
        </div>
    );
};

export default BottomCarousel;
