import React, {FunctionComponent} from 'react';

/* Types */
import {EHotelBadgeType, IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {prepareQaAttributes} from 'utilities/qaAttributes/qaAttributes';
import {getOperatorLinkPropsByParams} from 'projects/depreacted/hotels/utilities/getOperatorLinkPropsByParams/getOperatorLinkPropsByParams';

import * as i18nBlock from 'i18n/hotels-HotelPageOffer';

/* Components */
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import HotelOperator from 'projects/depreacted/hotels/components/HotelOperator/HotelOperator';
import HotelOfferLabels from 'projects/depreacted/hotels/components/HotelOfferLabels/HotelOfferLabels';
import HotelOfferBadges from 'projects/depreacted/hotels/components/HotelOfferBadges/HotelOfferBadges';
import {EFormatLabel} from 'projects/depreacted/hotels/components/HotelsCancellationInfo/HotelsCancellationInfo';

import {IBaseHotelPageOfferProps} from '../../HotelPageOffer';

/* Styles */
import cx from './HotelPageOfferDesktop.scss';

export interface IHotelPageOfferDesktopProps extends IBaseHotelPageOfferProps {}

/*
 * Render functions
 */
function renderLabels({
    offer,
    isColumnLabels,
    formatCancellationInfoLabel,
}: {
    offer: IHotelOffer;
    isColumnLabels?: boolean;
    formatCancellationInfoLabel?: EFormatLabel;
}): React.ReactNode {
    return (
        <div
            className={cx('labels', {
                labels_type_column: isColumnLabels,
            })}
            {...prepareQaAttributes('hotelOfferLabels')}
        >
            <HotelOfferLabels
                formatCancellationInfoLabel={formatCancellationInfoLabel}
                cancellationLabelClassName={cx('label')}
                mealLabelClassName={cx('label')}
                offer={offer}
            />
        </div>
    );
}

function renderOperator(
    offer: IHotelOffer,
    operatorById: OperatorByIdType,
    isBookOnYandex: boolean,
    theme: 'black' | 'ghost',
    onOfferSelect?: () => void,
    isHeadless?: boolean,
): React.ReactNode {
    const {operatorId, landingUrl} = offer;
    const size = theme === 'ghost' ? 'xs' : 'm';
    const align = theme === 'ghost' ? 'right' : 'left';
    const badgesAlign = theme === 'ghost' ? 'flex-end' : 'flex-start';

    const suitableBadges = isBookOnYandex
        ? offer.badges?.filter(({id}) => id !== EHotelBadgeType.HOTEL_DIRECT)
        : offer.badges;

    return (
        <div
            className={cx('partner')}
            {...prepareQaAttributes('hotelOfferPartner')}
        >
            <HotelOperator
                partnerUrl={landingUrl}
                operator={operatorById[operatorId]}
                theme={theme}
                size={size}
                align={align}
                onClick={onOfferSelect}
                isHeadless={isHeadless}
            />
            <HotelOfferBadges
                badges={suitableBadges}
                above={1}
                below={1}
                alignItems={badgesAlign}
            />
        </div>
    );
}

function renderPrice(
    offer: IHotelOffer,
    operatorById: OperatorByIdType,
    isBookOnYandex: boolean,
    isShortOperator?: boolean,
    onOfferSelect?: () => void,
    isHeadless?: boolean,
): React.ReactNode {
    const {price, landingUrl} = offer;

    return (
        <div className={cx('price')}>
            <div className={cx('price__value')}>
                <Price {...price} className={cx('price__operator')} />
                {isShortOperator &&
                    renderOperator(
                        offer,
                        operatorById,
                        isBookOnYandex,
                        'ghost',
                        onOfferSelect,
                        isHeadless,
                    )}
            </div>
            <ButtonLink
                theme="primary"
                size="m-inset"
                url={landingUrl}
                onClick={onOfferSelect}
                {...getOperatorLinkPropsByParams({isHeadless})}
                {...prepareQaAttributes('hotelOfferSelectButton')}
            >
                {i18nBlock.select()}
            </ButtonLink>
        </div>
    );
}

/** TODO: 💀 DeadCode - https://st.yandex-team.ru/TRAVELFRONT-6523 */
const HotelPageOfferDesktop: FunctionComponent<IHotelPageOfferDesktopProps> =
    props => {
        const {
            className,
            offer,
            isIe,
            operatorById,
            isColumnLabels,
            isShortOperator,
            isBookOnYandex,
            onOfferSelect,
            formatCancellationInfoLabel,
            isHeadless,
        } = props;
        const {id, name} = offer;

        return (
            <section
                className={cx(
                    'root',
                    {root_full: !isShortOperator, root_isIE: isIe},
                    className,
                )}
                {...prepareQaAttributes({
                    key: id,
                    parent: props,
                    current: 'hotelPageOffer',
                })}
            >
                <div
                    className={cx('offerName')}
                    {...prepareQaAttributes('hotelOfferName')}
                >
                    {name}
                </div>
                {renderLabels({
                    offer,
                    isColumnLabels,
                    formatCancellationInfoLabel,
                })}
                {!isShortOperator &&
                    renderOperator(
                        offer,
                        operatorById,
                        isBookOnYandex,
                        'black',
                        onOfferSelect,
                        isHeadless,
                    )}
                {renderPrice(
                    offer,
                    operatorById,
                    isBookOnYandex,
                    isShortOperator,
                    onOfferSelect,
                    isHeadless,
                )}
            </section>
        );
    };

export default HotelPageOfferDesktop;
