import React, {PureComponent, ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {IWithDeviceType} from 'types/withDeviceType';
import {IHotelOrRoomImage} from 'types/hotels/hotel/IHotelImages';
import {EHotelImageAltayStandardSize} from 'types/hotels/common/EHotelImageAltayStandardSize';

import {
    convertHotelOrRoomImages,
    convertUrlTemplateImagesToFlatList,
} from 'projects/depreacted/hotels/utilities/prepareAndParseImages/prepareAndParseImages';
import {deviceMods} from 'utilities/stylesUtils';

import ImagesCarousel from 'components/ImagesCarousel/ImagesCarousel';

import GalleryRow from '../Gallery/GalleryRow/GalleryRow';

/* Init styles */
import cx from './HotelPageRoomGallery.scss';

/* Constants */
const START_TILE_IMAGES_INDEX = 1;
const TOTAL_TILE_IMAGES = 5;
const TOTAL_GALLERY_IMAGES = START_TILE_IMAGES_INDEX + TOTAL_TILE_IMAGES;
const DESKTOP_IMAGES_SPACE_BETWEEN = 0;
const MOBILE_IMAGES_SPACE_BETWEEN = 1;

/* Types */
interface IHotelPageRoomGalleryProps extends IWithClassName, IWithDeviceType {
    images: IHotelOrRoomImage[];
    onImageClick?: (imageIndex: number) => void;
    imagesAlt?: string;
}

/** TODO: 💀 DeadCode - https://st.yandex-team.ru/TRAVELFRONT-6523 */
class HotelPageRoomGallery extends PureComponent<IHotelPageRoomGalleryProps> {
    static defaultProps = {
        images: [],
    };

    private handleCarouselImageClick = (src: string, index: number): void => {
        const {onImageClick} = this.props;

        if (onImageClick) {
            onImageClick(index);
        }
    };

    private handleRowImageClick = (index: number): void => {
        const {onImageClick} = this.props;

        if (onImageClick) {
            onImageClick(index + START_TILE_IMAGES_INDEX);
        }
    };

    private renderImagesCarousel(): ReactNode {
        const {images, deviceType} = this.props;
        const {isDesktop} = deviceType;
        const convertedImages = convertHotelOrRoomImages(
            images,
            EHotelImageAltayStandardSize.L,
        );

        return (
            <ImagesCarousel
                className={cx(
                    'imagesCarousel',
                    deviceMods('imagesCarousel', deviceType),
                )}
                imageClassName={cx('carouselImage')}
                type={isDesktop ? 'mini' : 'full'}
                arrowsType="dark"
                spaceBetween={
                    isDesktop
                        ? DESKTOP_IMAGES_SPACE_BETWEEN
                        : MOBILE_IMAGES_SPACE_BETWEEN
                }
                images={convertedImages}
                onImageClick={this.handleCarouselImageClick}
                isDesktop={isDesktop}
                autoHeight={isDesktop}
                lazyLoad
            />
        );
    }

    private renderTileOverlay = (): ReactNode => {
        const {images} = this.props;
        const totalImageCount = images.length;

        return (
            <div className={cx('totalTileOverlay')}>
                {`+${
                    totalImageCount -
                    TOTAL_TILE_IMAGES -
                    START_TILE_IMAGES_INDEX
                }`}
            </div>
        );
    };

    private renderTileImages(): ReactNode {
        const {images, imagesAlt} = this.props;
        const totalImageCount = images.length;
        const sizedImagesUrls = convertUrlTemplateImagesToFlatList(
            images,
            EHotelImageAltayStandardSize.S,
        ).slice(START_TILE_IMAGES_INDEX, TOTAL_GALLERY_IMAGES);

        return (
            <GalleryRow
                className={cx('tileImages')}
                imageContainerClassName={cx('tileImage')}
                clickableWrapperClassName={cx('tileImagesClickable')}
                imagesAlt={imagesAlt}
                isLastClickable={totalImageCount > TOTAL_GALLERY_IMAGES}
                imageSources={sizedImagesUrls}
                totalImageCount={totalImageCount}
                customRenderOverlay={this.renderTileOverlay}
                onImageClick={this.handleRowImageClick}
                onMoreImagesClick={this.handleRowImageClick}
            />
        );
    }

    render(): React.ReactNode {
        const {
            images,
            className,
            deviceType: {isDesktop},
        } = this.props;

        if (images.length > 0) {
            return (
                <div className={cx(className, 'roomGallery')}>
                    {this.renderImagesCarousel()}
                    {isDesktop && this.renderTileImages()}
                </div>
            );
        }

        return null;
    }
}

export default HotelPageRoomGallery;
