import {useEffect, useState} from 'react';

import {IHotelSlugRouteMatch} from 'types/hotels/hotel/IHotelSlugRouteMatch';
import {HotelSlugType, PermalinkType} from 'types/hotels/hotel/IHotel';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';
import {
    ReactRouterMatchType,
    ReactRouterLocationType,
} from 'types/common/IReactRouterLocation';

import {checkDifferenceBetweenRouterAndHotelIdentifier} from './utilities/checkDifferenceBetweenRouterAndHotelIdentifier';
import {checkDifferenceBetweenQueryAndSearchParams} from './utilities/checkDifferenceBetweenQueryAndSearchParams';
import {useDidMountEffect} from 'utilities/hooks/useDidMountEffect';
import getQueryByLocation from 'utilities/getQueryByLocation/getQueryByLocation';

import {useLastSearchTimeMarkerChange} from 'projects/depreacted/hotels/hooks/useLastSearchTimeMarkerChange';

/* Types */
export interface IHotelsSearchControllerProps {
    location: ReactRouterLocationType;
    match: ReactRouterMatchType<IHotelSlugRouteMatch>;
    hotelSlug?: HotelSlugType;
    permalink?: PermalinkType;
    searchParams?: IRequiredOfferParams;
    onChangeHotelIdentifier: () => void;
    onChangeLocationSearchParams: () => void;
    onChangeHotelInfoSearchParams: () => void;
    onStartFetchAdditionalInfo: () => void;
    onStopHotelInfoActions: () => void;
    onSearchParamsRemoved: () => void;
    resetHotelInfoToInitial: () => void;
    resetHotelSearchForm: () => void;
}

const HotelPageSearchController = (props: IHotelsSearchControllerProps) => {
    const {
        match,
        location,
        hotelSlug,
        permalink,
        searchParams,
        onChangeHotelIdentifier,
        onChangeLocationSearchParams,
        onChangeHotelInfoSearchParams,
        onStartFetchAdditionalInfo,
        resetHotelInfoToInitial,
        onSearchParamsRemoved,
        onStopHotelInfoActions,
    } = props;

    const [
        canEmitHotelInfoSearchParamsChangedEvent,
        allowEmitHotelInfoSearchParamsChangedEvent,
    ] = useState(false);

    /* 1. DidMount */
    useEffect(() => {
        if (hotelSlug) {
            onStartFetchAdditionalInfo();
        }
    }, []);

    /* 2. Location changed */
    const changedLastSearchTimeMarker = useLastSearchTimeMarkerChange(location);
    const componentMounted = useDidMountEffect();

    useEffect(() => {
        const {checkinDate, checkoutDate} = getQueryByLocation(location);

        /* Назад к странице отеля без дат */
        if (
            searchParams?.checkinDate &&
            searchParams?.checkoutDate &&
            !checkinDate &&
            !checkoutDate
        ) {
            onSearchParamsRemoved();

            return;
        }

        /* 2.1 Location hotelIdentifier */
        const canFetchHotelInfo =
            checkDifferenceBetweenRouterAndHotelIdentifier({
                match,
                location,
                hotelSlug,
                permalink,
            });

        if (canFetchHotelInfo) {
            onChangeHotelIdentifier();
        } else {
            /* 2.2 Location searchParams */
            const needSyncSearchParams =
                checkDifferenceBetweenQueryAndSearchParams({
                    location,
                    searchParams,
                });

            if (
                componentMounted &&
                (needSyncSearchParams || changedLastSearchTimeMarker)
            ) {
                onChangeLocationSearchParams();
                allowEmitHotelInfoSearchParamsChangedEvent(true);
            }
        }
    }, [location]);

    /* 3. HotelInfo searchParams changed */
    useEffect(() => {
        if (searchParams && canEmitHotelInfoSearchParamsChangedEvent) {
            onChangeHotelInfoSearchParams();
            allowEmitHotelInfoSearchParamsChangedEvent(false);
        }
    }, [searchParams]);

    /* 4. WillUnMount */
    useEffect(() => {
        return () => {
            onStopHotelInfoActions();
            resetHotelInfoToInitial();
        };
    }, [resetHotelInfoToInitial]);

    return null;
};

export default HotelPageSearchController;
