import React, {
    Component,
    ReactNode,
    RefObject,
    createRef,
    MutableRefObject,
} from 'react';

import {
    BOY_META_MAIN_TAB_MAX_LIMITS,
    BOY_META_MAIN_TAB_MAX_OFFERS,
    DEFAULT_ADULTS_COUNT,
} from 'constants/hotels';

import {IWithDeviceType} from 'types/withDeviceType';
import {
    IHotelOffer,
    IRequiredOfferParams,
} from 'types/hotels/offer/IHotelOffer';
import {
    FilterType,
    HotelOffersFiltersType,
    IHotelOffersInfo,
} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';
import {
    ESearchFormFieldName,
    ESearchFormSize,
} from 'components/SearchForm/types';
import {IWithClassName} from 'types/withClassName';

import {deviceMods} from 'utilities/stylesUtils';
import humanizePeriod from 'utilities/dateUtils/humanizePeriod';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {getTotalNights} from 'projects/depreacted/hotels/utilities/calculateTotalNights/calculateTotalNights';
import getGuestsCount from 'projects/depreacted/hotels/pages/HotelPage/utilities/getGuestsCount';
import scrollToNode from 'utilities/dom/scrollToNode';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import Heading from 'components/Heading/Heading';
import Card from 'components/Card/Card';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import Text from 'components/Text/Text';
import Box from 'components/Box/Box';
import MainOffers from './components/MainOffers/MainOffers';
import PartnerOffers from './components/PartnerOffers/PartnerOffers';
import MainOffersSkeleton from './components/MainOffersSkeleton/MainOffersSkeleton';
import PartnerOffersSkeleton from './components/PartnerOffersSkeleton/PartnerOffersSkeleton';
import HotelPageSearchForm from 'projects/depreacted/hotels/pages/HotelPage/components/HotelPageSearchForm/HotelPageSearchForm';
import Rooms from 'projects/depreacted/hotels/pages/HotelPage/components/OffersInfo/components/Rooms/Rooms';
import Flex from 'components/Flex/Flex';
import {IFormFieldsRef} from 'components/SearchForm/SearchForm';

import SpecifyDatesWarning from '../SpecifyDatesWarning/SpecifyDatesWarning';

import cx from './OffersInfo.scss';

interface IHotelOfferInfoProps
    extends IWithClassName,
        IWithDeviceType,
        IWithQaAttributes {
    mainOffersClassName?: string;
    partnerOffersClassName?: string;
    searchParams?: IRequiredOfferParams;
    isLoading?: boolean;
    offersInfo?: IHotelOffersInfo;
    filteredOffers?: IHotelOffer[];
    isRoomGroup?: boolean;
    isFullOffersTab?: boolean;
    filters?: HotelOffersFiltersType;
    onFilterSelect?: (filter: FilterType) => void;
    onOfferSelect?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onAllOffersClick?: () => void;
    backButtonLink?: string;
    chosenId?: string;
    roomsRef?: React.RefObject<HTMLDivElement>;
    withRoomsMatching?: boolean;
}

class OffersInfo extends Component<IHotelOfferInfoProps> {
    searchFormRef: RefObject<HTMLFormElement> = createRef();
    searchFormFieldsRef: MutableRefObject<IFormFieldsRef | undefined> = {
        current: undefined,
    };

    private getPartnerOffersTitle(hasMainOffers: boolean): string {
        const {searchParams} = this.props;

        if (hasMainOffers || !searchParams) {
            return i18nBlock.partnerTitleShort();
        }

        const {
            checkinDate,
            checkoutDate,
            adults,
            childrenAges: {length: children},
        } = searchParams;

        return i18nBlock.partnerTitleDetailed({
            dates: humanizePeriod(checkinDate, checkoutDate),
            adults,
            children,
        });
    }

    private handleCheckPricesClick = (): void => {
        if (this.searchFormRef.current) {
            scrollToNode(this.searchFormRef.current, {
                shiftY: -120,
                behavior: 'smooth',
            });
        }

        if (this.searchFormFieldsRef.current) {
            this.searchFormFieldsRef.current.focusFieldByName(
                ESearchFormFieldName.START_DATE,
            );
        }
    };

    private getTotalNights(): number | undefined {
        const {searchParams} = this.props;

        return searchParams
            ? getTotalNights(
                  searchParams.checkinDate,
                  searchParams.checkoutDate,
              )
            : undefined;
    }

    private renderAllOffersLink(): ReactNode {
        const {
            onAllOffersClick,
            offersInfo,
            deviceType: {isMobile},
        } = this.props;

        const offerCount = offersInfo?.mainOffers?.length;

        if (
            onAllOffersClick &&
            offerCount &&
            offerCount >
                BOY_META_MAIN_TAB_MAX_OFFERS + BOY_META_MAIN_TAB_MAX_LIMITS
        ) {
            return (
                <div className={cx('showMore')}>
                    <ButtonLink
                        onClick={onAllOffersClick}
                        {...prepareQaAttributes('allRoomsLink')}
                        theme="primary"
                        size="l"
                        width={isMobile ? 'max' : 'auto'}
                    >
                        {i18nBlock.allOffersWithCount({offerCount})}
                    </ButtonLink>
                </div>
            );
        }

        return null;
    }

    private renderPartnerOffers(hasMainOffers: boolean): ReactNode {
        const {offersInfo, deviceType, onOfferSelect} = this.props;
        const {isMobile} = deviceType;

        if (offersInfo) {
            const {partnerOffers = [], operatorById} = offersInfo;
            const nightsCount = this.getTotalNights();

            const offersNode = (
                <PartnerOffers
                    className={cx('offers')}
                    hasMainOffers={hasMainOffers}
                    offers={partnerOffers}
                    operatorById={operatorById}
                    onOfferSelect={onOfferSelect}
                    nightsCount={nightsCount}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'partnerOffers',
                    })}
                />
            );

            return isMobile ? (
                <Card shadow="default">{offersNode}</Card>
            ) : (
                offersNode
            );
        }

        return null;
    }

    renderOffers(): ReactNode {
        const {
            offersInfo,
            isFullOffersTab,
            deviceType,
            onOfferSelect,
            isRoomGroup = false,
            onOfferWatchButtonClick,
            chosenId,
        } = this.props;
        const {isMobile} = deviceType;

        if (offersInfo) {
            const {mainOffers = [], partnerOffers, operatorById} = offersInfo;
            const nightsCount = this.getTotalNights();

            const offersNode = (
                <>
                    <MainOffers
                        className={cx('offers')}
                        deviceType={deviceType}
                        offers={mainOffers}
                        partnerOffers={partnerOffers}
                        operatorById={operatorById}
                        onOfferSelect={onOfferSelect}
                        isFullOffersTab={isFullOffersTab}
                        nightsCount={nightsCount}
                        isRoomGroup={isRoomGroup}
                        onOfferWatchButtonClick={onOfferWatchButtonClick}
                        chosenId={chosenId}
                        {...prepareQaAttributes({
                            parent: this.props,
                            current: 'mainOffers',
                        })}
                    />
                </>
            );

            return isMobile ? (
                <>
                    <Card shadow="default">{offersNode}</Card>
                    {!isFullOffersTab && this.renderAllOffersLink()}
                </>
            ) : (
                <>
                    {offersNode}
                    {!isFullOffersTab && this.renderAllOffersLink()}
                </>
            );
        }

        return null;
    }

    renderRooms(): ReactNode {
        const {
            offersInfo,
            onOfferSelect,
            searchParams,
            onAllOffersClick,
            onOfferWatchButtonClick,
            chosenId,
        } = this.props;

        if (offersInfo && offersInfo.rooms) {
            const nightsCount = this.getTotalNights();
            const canRenderRoomImage = offersInfo.rooms.some(
                room => room.images && room.images.length > 0,
            );

            return (
                <Rooms
                    offersInfo={offersInfo}
                    onOfferSelect={onOfferSelect}
                    canShowImages={canRenderRoomImage}
                    nightsCount={nightsCount}
                    isShortView={false}
                    onAllOffersClick={onAllOffersClick}
                    onOfferWatchButtonClick={onOfferWatchButtonClick}
                    onCheckPricesClick={this.handleCheckPricesClick}
                    chosenRoomId={chosenId}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'rooms',
                    })}
                    searchParams={searchParams}
                />
            );
        }

        return null;
    }

    renderWithoutDatesPageTitle(): ReactNode {
        return (
            <Flex alignItems="center" between={3} inline>
                <Heading
                    level={2}
                    className={cx('titleWithoutDates')}
                    {...prepareQaAttributes('offersTitle')}
                >
                    {i18nBlock.availabilityOfRooms()}
                </Heading>
                <SpecifyDatesWarning />
            </Flex>
        );
    }

    renderMainTitle(): ReactNode {
        const {searchParams} = this.props;

        if (!searchParams) {
            return this.renderWithoutDatesPageTitle();
        }

        const {
            checkinDate,
            checkoutDate,
            adults,
            childrenAges: {length: children},
        } = searchParams;

        return (
            <Heading
                level={2}
                className={cx('title')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'mainOffersTitle',
                })}
            >
                {i18nBlock.heading({
                    dates: humanizePeriod(checkinDate, checkoutDate),
                    adults,
                    children,
                })}
            </Heading>
        );
    }

    renderPartnerTitle(hasMainOffers: boolean): ReactNode {
        const title = this.getPartnerOffersTitle(hasMainOffers);

        return (
            <Heading
                level={2}
                className={cx('title')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'partnerOffersTitle',
                })}
            >
                {title}
            </Heading>
        );
    }

    private renderFilteredOffersEmpty(): ReactNode {
        return (
            <div
                className={cx('empty')}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'filteredOffersEmpty',
                })}
            >
                <Text size="m" weight="medium" className={cx('textEmpty')}>
                    {i18nBlock.emptyDotFiltersDotFirstLine()}
                </Text>
                <Text size="m" className={cx('textEmpty')}>
                    {i18nBlock.emptyDotFiltersDotSecondLine()}
                </Text>
            </div>
        );
    }

    private renderEmptyOffers(): ReactNode {
        const {deviceType, backButtonLink, searchParams} = this.props;
        const guestsCount = searchParams
            ? getGuestsCount(searchParams)
            : DEFAULT_ADULTS_COUNT;

        return (
            <div
                className={cx('empty', deviceMods('empty', deviceType))}
                {...prepareQaAttributes({
                    parent: this.props,
                    current: 'emptyOffers',
                })}
            >
                <Box below={4}>
                    <Text size="m" weight="medium" className={cx('textEmpty')}>
                        {i18nBlock.emptyFirstLineNew({guestsCount})}
                    </Text>
                    <Text size="m" className={cx('textEmpty')}>
                        {i18nBlock.emptySecondLineNew()}
                    </Text>
                </Box>
                <ButtonLink
                    theme="secondary"
                    size={deviceType.isMobile ? 'l' : 'm-inset'}
                    to={backButtonLink}
                >
                    {i18nBlock.emptyButtonNew()}
                </ButtonLink>
            </div>
        );
    }

    private renderControls(): ReactNode {
        const {deviceType, isFullOffersTab, searchParams} = this.props;
        const withoutDates = !searchParams;

        return (
            <div
                className={cx('searchForm', {
                    searchForm_gray: isFullOffersTab,
                    searchForm_withoutDates: withoutDates,
                })}
                {...prepareQaAttributes('hotelPageSearchForm')}
            >
                <HotelPageSearchForm
                    deviceType={deviceType}
                    size={ESearchFormSize.M}
                    submitButtonTheme={withoutDates ? 'primary' : 'secondary'}
                    submitButtonText={
                        withoutDates ? i18nBlock.checkPrices() : undefined
                    }
                    formRef={this.searchFormRef}
                    fieldsRef={this.searchFormFieldsRef}
                    {...prepareQaAttributes({
                        parent: this.props,
                        current: 'hotelPageSearchForm',
                    })}
                />
            </div>
        );
    }

    private renderFlatOfferListWithFilters(): ReactNode {
        const {
            offersInfo = {mainOffers: []},
            filteredOffers = [],
            filters = {},
        } = this.props;
        const {mainOffers = []} = offersInfo;

        const hasOffersToShow = mainOffers.length > 0;
        const hasFiltersAndOffers =
            Object.keys(filters).length > 0 ? filteredOffers.length > 0 : true;

        if (hasOffersToShow) {
            return (
                <>
                    {hasFiltersAndOffers
                        ? this.renderOffers()
                        : this.renderFilteredOffersEmpty()}
                </>
            );
        }

        return null;
    }

    private renderContentByGroupType(): ReactNode {
        const {offersInfo} = this.props;

        const hasRooms = Boolean(offersInfo?.rooms?.length);

        if (hasRooms) {
            return this.renderRooms();
        }

        return this.renderFlatOfferListWithFilters();
    }

    private renderContent(): ReactNode {
        const {
            offersInfo,
            deviceType,
            isFullOffersTab,
            roomsRef,
            mainOffersClassName,
            partnerOffersClassName,
            isLoading,
            searchParams,
            withRoomsMatching,
        } = this.props;
        const isFinished =
            offersInfo?.offerSearchProgress?.finished && !isLoading;

        const hasMainOffers = Boolean(offersInfo?.mainOffers?.length);
        const hasPartnerOffers = Boolean(offersInfo?.partnerOffers?.length);
        const hasRooms = Boolean(offersInfo?.rooms?.length);

        const renderMainOffers = isFinished && hasMainOffers;
        const renderPartnerOffers =
            isFinished && hasPartnerOffers && !(hasRooms && withRoomsMatching);
        const isWithoutDatesPage = isFinished && !searchParams && hasRooms;
        const noOffers = isFinished && !hasMainOffers && !hasPartnerOffers;

        if (!isFinished) {
            return (
                <>
                    <div
                        className={cx(
                            'root',
                            deviceMods('root', deviceType),
                            mainOffersClassName,
                        )}
                    >
                        {this.renderMainTitle()}
                        {this.renderControls()}
                        <MainOffersSkeleton isFullOffersTab={isFullOffersTab} />
                    </div>
                    <div
                        className={cx(
                            'root',
                            deviceMods('root', deviceType),
                            partnerOffersClassName,
                        )}
                    >
                        {this.renderPartnerTitle(hasMainOffers)}
                        <PartnerOffersSkeleton />
                    </div>
                </>
            );
        }

        if (isWithoutDatesPage) {
            return (
                <div
                    ref={roomsRef}
                    className={cx(
                        'root',
                        deviceMods('root', deviceType),
                        mainOffersClassName,
                    )}
                >
                    {this.renderWithoutDatesPageTitle()}
                    {this.renderControls()}
                    {this.renderRooms()}
                </div>
            );
        }

        if (noOffers) {
            return (
                <div
                    ref={roomsRef}
                    className={cx(
                        'root',
                        deviceMods('root', deviceType),
                        mainOffersClassName,
                    )}
                >
                    {this.renderMainTitle()}
                    {this.renderControls()}
                    {Boolean(searchParams) && this.renderEmptyOffers()}
                </div>
            );
        }

        return (
            <>
                {renderMainOffers && (
                    <div
                        ref={roomsRef}
                        className={cx(
                            'root',
                            deviceMods('root', deviceType),
                            mainOffersClassName,
                        )}
                        {...prepareQaAttributes('hotelMainOffers')}
                    >
                        {this.renderMainTitle()}
                        {this.renderControls()}
                        {this.renderContentByGroupType()}
                    </div>
                )}
                {renderPartnerOffers && (
                    <div
                        ref={renderMainOffers ? null : roomsRef}
                        className={cx(
                            'root',
                            deviceMods('root', deviceType),
                            partnerOffersClassName,
                        )}
                        {...prepareQaAttributes('hotelPartnerOffers')}
                    >
                        {this.renderPartnerTitle(hasMainOffers)}
                        {!renderMainOffers && this.renderControls()}
                        {this.renderPartnerOffers(hasMainOffers)}
                    </div>
                )}
            </>
        );
    }

    render(): ReactNode {
        const {className} = this.props;

        return (
            <div className={className} {...prepareQaAttributes(this.props)}>
                {this.renderContent()}
            </div>
        );
    }
}

export default OffersInfo;
