import {FunctionComponent, ReactNode} from 'react';

import {IHotelOffer} from 'types/hotels/offer/IHotelOffer';
import {OperatorByIdType} from 'types/hotels/offer/IHotelsOperator';

import {useBoolean} from 'utilities/hooks/useBoolean';
import {insertJSXIntoKey} from 'utilities/tanker/insertJSXIntoKey';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/common-price';

import HotelOfferLabels from 'projects/depreacted/hotels/components/HotelOfferLabels/HotelOfferLabels';
import Price from 'components/Price/Price';
import RotatingArrowIcon from 'components/RotatingArrowIcon/RotatingArrowIcon';
import AutoHeightAnimateControl from 'components/AutoHeightAnimateControl/AutoHeightAnimateControl';
import MatchedOffer from './components/MatchedOffer/MatchedOffer';
import YandexOffer from './components/YandexOffer/YandexOffer';
import Flex from 'components/Flex/Flex';

import cx from './MatchedOffers.scss';

interface IMatchedOffersProps {
    mainOffer: IHotelOffer;
    offers: IHotelOffer[];
    operatorById: OperatorByIdType;
    isLast: boolean;
    hotelsPercentDiscount?: boolean;
}

const MatchedOffers: FunctionComponent<IMatchedOffersProps> = ({
    mainOffer,
    offers,
    operatorById,
    isLast,
    hotelsPercentDiscount,
}) => {
    const needAccordion = offers.length > 1;
    const deviceType = useDeviceType();
    const {isMobile} = deviceType;
    const {value: isExpanded, toggle: toggleExpanded} = useBoolean(
        !needAccordion,
    );

    const renderOffers = (): ReactNode => (
        <div className={cx('list')}>
            {offers.map(o =>
                o.yandexOffer ? (
                    <YandexOffer
                        key="yandex"
                        className={cx('offer')}
                        offer={o}
                        withHighlight={offers.length > 1}
                        hotelsPercentDiscount={hotelsPercentDiscount}
                    />
                ) : (
                    <MatchedOffer
                        key={o.operatorId}
                        className={cx('offer')}
                        operator={operatorById[o.operatorId]}
                        partnerUrl={o.landingUrl}
                        price={o.price}
                    />
                ),
            )}
        </div>
    );

    return (
        <div
            className={cx('root', deviceMods('root', deviceType), {
                root_expanded: isExpanded,
                root_last: isLast,
            })}
        >
            <Flex
                className={cx('header')}
                alignItems={isMobile ? 'flex-start' : 'center'}
                flexDirection={isMobile ? 'column' : 'row'}
                onClick={needAccordion ? toggleExpanded : undefined}
            >
                <div className={cx('labels')}>
                    <HotelOfferLabels
                        offer={mainOffer}
                        cancellationLabelClassName={cx('cancellationLabel')}
                        isMealIconHidden={true}
                        shortCancellationDate
                    />
                </div>
                <span className={cx('price')}>
                    {insertJSXIntoKey(i18nBlock._from)({
                        price: <Price {...mainOffer.price} />,
                    })}
                </span>

                {needAccordion && (
                    <RotatingArrowIcon
                        rotated={isExpanded}
                        className={cx('arrow')}
                    />
                )}
            </Flex>
            {needAccordion ? (
                <AutoHeightAnimateControl canShowComponent={isExpanded}>
                    {renderOffers()}
                </AutoHeightAnimateControl>
            ) : (
                renderOffers()
            )}
        </div>
    );
};

export default MatchedOffers;
