import React, {ReactNode, useMemo} from 'react';

import {MIN_PERCENT_TO_SHOW} from 'projects/depreacted/hotels/constants/discount';

import {
    IDiscountInfo,
    EHotelBadgeType,
    IHotelOrOfferBadge,
    IOfferYandexPlusInfo,
} from 'types/hotels/offer/IHotelOffer';
import {IWithClassName} from 'types/withClassName';
import {TLinkTarget} from 'types/common/TLinkTarget';

import IPrice from 'utilities/currency/PriceInterface';
import {getOperatorLinkPropsByParams} from 'projects/depreacted/hotels/utilities/getOperatorLinkPropsByParams/getOperatorLinkPropsByParams';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceModDesktop, deviceMods} from 'utilities/stylesUtils';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import Text from 'components/Text/Text';
import Price from 'components/Price/Price';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import HotelOfferBadges from 'projects/depreacted/hotels/components/HotelOfferBadges/HotelOfferBadges';
import Flex, {IFlexProps} from 'components/Flex/Flex';
import PlusInfoPopup from 'components/PlusInfoPopup/PlusInfoPopup';
import YandexEdaPromo from 'projects/depreacted/hotels/components/YandexEdaPromo/YandexEdaPromo';
import DiscountInfo from 'projects/depreacted/hotels/components/HotelCard/components/DiscountInfo/DiscountInfo';

import cx from './Offer.scss';

export interface IOfferCollapsed
    extends IWithClassName,
        IWithQaAttributes,
        Pick<IFlexProps, 'justifyContent'> {
    labels: React.ReactElement;
    price: IPrice;
    discountInfo?: IDiscountInfo;
    hotelsPercentDiscount?: boolean;
    buttonTheme: 'secondary' | 'primary';
    buttonText: string;
    showNightsCount: boolean;
    nightsCount?: number;
    title?: React.ReactElement | string;
    onOfferSelect?: () => void;
    buttonUrl?: string;
    isHeadless?: boolean;
    badges?: IHotelOrOfferBadge[];
    type?: 'compact' | 'horizontal';
    isWide?: boolean;
    isPartner?: boolean;
    plusInfo?: IOfferYandexPlusInfo;
    linkTarget?: TLinkTarget;
}

const Offer: React.FC<IOfferCollapsed> = props => {
    const {
        className,
        title,
        labels,
        isHeadless,
        onOfferSelect,
        price,
        discountInfo,
        justifyContent = 'flex-start',
        buttonTheme,
        buttonUrl,
        buttonText,
        nightsCount,
        badges,
        type = 'compact',
        isWide,
        showNightsCount,
        isPartner,
        plusInfo,
        linkTarget,
        hotelsPercentDiscount,
    } = props;
    const deviceType = useDeviceType();
    const {isMobile, isDesktop} = deviceType;
    const yandexEdaBadge = badges?.find(
        badge => badge.id === EHotelBadgeType.YANDEX_EDA,
    );
    const showDiscountPercent =
        hotelsPercentDiscount &&
        discountInfo?.percent &&
        discountInfo.percent >= MIN_PERCENT_TO_SHOW;

    const renderBadges = (): ReactNode => {
        if (!badges?.length) {
            return null;
        }

        return (
            <HotelOfferBadges
                badges={badges}
                above={type === 'horizontal' ? 1 : 4}
                below={2}
                alignItems={isMobile ? 'flex-start' : 'flex-end'}
            />
        );
    };

    const plusInfoBlock = useMemo(() => {
        return (
            plusInfo?.points && (
                <PlusInfoPopup
                    className={cx(
                        'plusOffer',
                        deviceModDesktop('plusOffer', deviceType),
                    )}
                    {...plusInfo}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'plusInfo',
                    })}
                />
            )
        );
    }, [deviceType, plusInfo]);

    return (
        <section
            className={cx(className, deviceMods('root', deviceType), {
                root_horizontal: type === 'horizontal',
                root_wide: isWide,
            })}
            {...prepareQaAttributes(props)}
        >
            {title && (
                <div
                    className={cx('offerName')}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'offerName',
                    })}
                >
                    {title}
                </div>
            )}
            <div
                className={cx('labels')}
                {...prepareQaAttributes({
                    parent: props,
                    current: 'labels',
                })}
            >
                {labels}
            </div>
            {type !== 'horizontal' && renderBadges()}
            <div
                className={cx('price', {
                    price_short: !showNightsCount,
                })}
            >
                <div className={cx('priceNights')}>
                    <Flex
                        alignItems="center"
                        justifyContent={justifyContent}
                        flexWrap="wrap"
                        between="2"
                        inline
                    >
                        <Price
                            className={cx('price__value')}
                            {...price}
                            color={
                                discountInfo?.strikethroughPrice &&
                                !showDiscountPercent
                                    ? 'alert'
                                    : undefined
                            }
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'price',
                            })}
                        />
                        {discountInfo && (
                            <DiscountInfo
                                discountInfo={discountInfo}
                                hotelsPercentDiscount={hotelsPercentDiscount}
                                percentSize={deviceType.isDesktop ? 's' : 'm'}
                            />
                        )}
                    </Flex>
                    {showNightsCount && nightsCount && (
                        <Text
                            size="s-inset"
                            className={cx('price__nights')}
                            {...prepareQaAttributes({
                                parent: props,
                                current: 'nightsCount',
                            })}
                        >
                            {i18nBlock.nightsCount({nightsCount})}
                        </Text>
                    )}
                    {isDesktop && plusInfoBlock}
                    {isDesktop && yandexEdaBadge && (
                        <YandexEdaPromo badge={yandexEdaBadge} />
                    )}
                    {type === 'horizontal' && renderBadges()}
                </div>
                <ButtonLink
                    theme={buttonTheme}
                    size={isMobile || (!isPartner && !isWide) ? 'l' : 'm-inset'}
                    url={buttonUrl}
                    onClick={onOfferSelect}
                    target={linkTarget}
                    {...getOperatorLinkPropsByParams({isHeadless})}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'bookButton',
                    })}
                >
                    {buttonText}
                </ButtonLink>
            </div>
            {isMobile && plusInfoBlock}
            {isMobile && yandexEdaBadge && (
                <div>
                    <YandexEdaPromo badge={yandexEdaBadge} />
                </div>
            )}
        </section>
    );
};

export default Offer;
