import React, {ReactNode} from 'react';

import {IWithClassName} from 'types/withClassName';
import {EPansionAggregate} from 'types/hotels/offer/IHotelOfferMeal';
import {ECancellationInfoAggregate} from 'types/hotels/offer/IHotelOfferCancellationInfo';

import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {deviceMods} from 'utilities/stylesUtils';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nBlock from 'i18n/hotels-HotelPageOffersInfo';

import TextWithIcon from 'components/TextWithIcon/TextWithIcon';
import AllInclusiveIcon from 'icons/16/AllInclusive';
import IncludeIcon from 'icons/16/Include';
import ExcludeIcon from 'icons/16/Exclude';

import cx from './OfferLabels.scss';

interface IOfferLabels extends IWithClassName, IWithQaAttributes {
    mealType?: EPansionAggregate | null;
    cancellationType?: ECancellationInfoAggregate | null;
    isWide?: boolean;
}

const OfferLabels: React.FC<IOfferLabels> = props => {
    const {className, mealType, cancellationType, isWide} = props;
    const deviceType = useDeviceType();

    const renderMealInfo = (): ReactNode => {
        if (mealType !== EPansionAggregate.PANSION_AVAILABLE) {
            if (!isWide) {
                return null;
            }

            return <div className={cx('label', 'mealInfo')} />;
        }

        return (
            <TextWithIcon
                size="m"
                className={cx('label', 'label_green', 'mealInfo')}
                iconLeft={AllInclusiveIcon}
                text={i18nBlock.mealVariants()}
                {...prepareQaAttributes({parent: props, current: 'mealInfo'})}
            />
        );
    };

    const renderCancellationInfo = (): ReactNode => {
        if (
            mealType === EPansionAggregate.PANSION_AVAILABLE &&
            cancellationType !==
                ECancellationInfoAggregate.FULLY_REFUNDABLE_AVAILABLE
        ) {
            return null;
        }

        if (
            cancellationType ===
            ECancellationInfoAggregate.FULLY_REFUNDABLE_AVAILABLE
        ) {
            return (
                <TextWithIcon
                    className={cx('label', 'label_green')}
                    size="m"
                    iconLeft={IncludeIcon}
                    text={i18nBlock.freeCancellationVariants()}
                    {...prepareQaAttributes({
                        parent: props,
                        current: 'cancellationInfo',
                    })}
                />
            );
        }

        return (
            <TextWithIcon
                className={cx('label')}
                size="m"
                iconLeft={ExcludeIcon}
                text={
                    cancellationType ===
                    ECancellationInfoAggregate.REFUNDABLE_WITH_PENALTY_AVAILABLE
                        ? i18nBlock.cancellationWithPenaltyVariants()
                        : i18nBlock.noCancellationVariants()
                }
                {...prepareQaAttributes({
                    parent: props,
                    current: 'cancellationInfo',
                })}
            />
        );
    };

    return (
        <div
            className={cx(deviceMods('root', deviceType), className, {
                root_wide: isWide,
            })}
        >
            {renderMealInfo()}
            {renderCancellationInfo()}
        </div>
    );
};

export default OfferLabels;
