import {FunctionComponent, ReactNode, useMemo} from 'react';
import {useSelector} from 'react-redux';

import {
    BOY_META_MAIN_TAB_MAX_ROOMS,
    BOY_META_MAIN_TAB_MAX_LIMITS,
} from 'constants/hotels';

import {IWithClassName} from 'types/withClassName';
import {
    IHotelBoyMetaRoomWithOffers,
    IHotelOffersInfo,
} from 'reducers/depreacted/hotels/hotelPage/hotelInfo/types';
import {IRequiredOfferParams} from 'types/hotels/offer/IHotelOffer';

import hotelRoomsBoyMetaSelector from 'selectors/depreacted/hotels/hotel/common/hotelRoomsBoyMetaSelector';

import {deviceModMobile, deviceMods} from 'utilities/stylesUtils';
import {useDeviceType} from 'utilities/hooks/useDeviceType';
import {
    IWithQaAttributes,
    prepareQaAttributes,
} from 'utilities/qaAttributes/qaAttributes';

import * as i18nNew from 'i18n/hotels-HotelPageRooms';

/* Components */
import Room from './components/Room/Room';
import {CardWithDeviceLayout} from 'components/CardWithDeviceLayout/CardWithDeviceLayout';
import ButtonLink from 'components/ButtonLink/ButtonLink';
import RoomsWithoutOffers from './components/RoomsWithoutOffers/RoomsWithoutOffers';

import cx from './Rooms.scss';

/* Component Types */
interface IHotelPageRoomsProps extends IWithClassName, IWithQaAttributes {
    offersInfo: IHotelOffersInfo;
    nightsCount?: number;
    /** TODO: 💀 DeadCode - https://st.yandex-team.ru/TRAVELFRONT-6523 */
    isShortView: boolean;
    canShowImages?: boolean;
    onOfferSelect?: () => void;
    onAllOffersClick?: () => void;
    onOfferWatchButtonClick?: (roomId: string) => void;
    onCheckPricesClick?: () => void;
    chosenRoomId?: string;
    searchParams?: IRequiredOfferParams;
}

const Rooms: FunctionComponent<IHotelPageRoomsProps> = ({
    onOfferSelect,
    canShowImages = true,
    nightsCount,
    className,
    isShortView,
    onAllOffersClick,
    onOfferWatchButtonClick,
    onCheckPricesClick,
    chosenRoomId,
    searchParams,
    offersInfo,
    ...rest
}) => {
    const hotelRooms = useSelector(hotelRoomsBoyMetaSelector);
    const deviceType = useDeviceType();
    const {partnerOffers, operatorById} = offersInfo;

    const isMoreRoomsLimits =
        isShortView &&
        hotelRooms.length >
            BOY_META_MAIN_TAB_MAX_ROOMS + BOY_META_MAIN_TAB_MAX_LIMITS;

    const rooms = useMemo(() => {
        if (isMoreRoomsLimits) {
            return hotelRooms.slice(0, BOY_META_MAIN_TAB_MAX_ROOMS);
        }

        return hotelRooms;
    }, [isMoreRoomsLimits, hotelRooms]);

    const {roomsWithOffers, roomsWithoutOffers} = useMemo(
        () =>
            rooms.reduce<{
                roomsWithOffers: IHotelBoyMetaRoomWithOffers[];
                roomsWithoutOffers: IHotelBoyMetaRoomWithOffers[];
            }>(
                (result, room) => {
                    if (room.offers.length) {
                        result.roomsWithOffers.push(room);
                    } else {
                        result.roomsWithoutOffers.push(room);
                    }

                    return result;
                },
                {roomsWithOffers: [], roomsWithoutOffers: []},
            ),
        [rooms],
    );

    const renderRoom = (room: IHotelBoyMetaRoomWithOffers): ReactNode => {
        return (
            <Room
                key={room.id}
                room={room}
                partnerOffers={partnerOffers}
                operatorById={operatorById}
                onOfferSelect={onOfferSelect}
                onOfferWatchButtonClick={onOfferWatchButtonClick}
                onCheckPricesClick={onCheckPricesClick}
                canShowImages={canShowImages}
                nightsCount={nightsCount}
                chosenRoomId={chosenRoomId}
                isShort={isShortView}
                searchParams={searchParams}
                {...prepareQaAttributes({
                    parent: rest,
                    current: 'room',
                    key: room.id,
                })}
            />
        );
    };

    const renderMore = (): ReactNode => {
        if (!isMoreRoomsLimits) {
            return null;
        }

        return (
            <div className={cx('showMore')}>
                <ButtonLink
                    onClick={onAllOffersClick}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'allRoomsLink',
                    })}
                    theme="primary"
                    size="l"
                    width={deviceType.isMobile ? 'max' : 'auto'}
                >
                    {i18nNew.moreRooms({roomCount: hotelRooms.length})}
                </ButtonLink>
            </div>
        );
    };

    if (isShortView && deviceType.isMobile) {
        return (
            <>
                <CardWithDeviceLayout
                    className={cx(
                        'root',
                        className,
                        deviceModMobile('root', deviceType),
                    )}
                >
                    {rooms.map(renderRoom)}
                </CardWithDeviceLayout>
                {renderMore()}
            </>
        );
    }

    const rootClassName = cx(
        className,
        'root',
        deviceMods('root', deviceType),
        {
            root_short: isShortView,
        },
    );

    if (!searchParams) {
        return (
            <div className={rootClassName}>
                {roomsWithoutOffers.map(renderRoom)}
            </div>
        );
    }

    return (
        <>
            <div className={rootClassName}>
                {roomsWithOffers.map(renderRoom)}
            </div>

            {Boolean(roomsWithoutOffers.length) && (
                <RoomsWithoutOffers
                    className={rootClassName}
                    checkinDate={searchParams.checkinDate}
                    checkoutDate={searchParams.checkoutDate}
                    rooms={roomsWithoutOffers}
                    renderRoom={renderRoom}
                    {...prepareQaAttributes({
                        parent: rest,
                        current: 'roomsWithoutOffers',
                    })}
                />
            )}

            {renderMore()}
        </>
    );
};

export default Rooms;
